/*!
	@class		BluePhone
	@brief		Hauptklasse von BluePhone.
	@author 	Thomas Gemperli, <bluephone@gemperli.net>
	@version	1.0
	@date		2004-08-03
	@par		This program is free software; you can redistribute it and/or 
				modify it under the terms of the GNU General Public License.
	@file		bluephone.cpp
*/


#include "bluephone.h"

#include <qlineedit.h>
#include <qprogressbar.h>
#include <qradiobutton.h> 
#include <qspinbox.h>
#include <qmessagebox.h>
#include <qlistview.h>
#include <qtabwidget.h>
#include <qfiledialog.h>
#include <kled.h>
#include <kfileitem.h>
#include <fstream>
#include <unistd.h>



/**
  * BluePhone Konstruktor
  * Erstellt das QWidget Objekt BluePhone im Hauptfenster.
  */
BluePhone::BluePhone(QWidget* parent, const char* name, WFlags fl)
		: FormBluePhone(parent,name,fl)
{

	/* Initialisierungen. */
	version = "1.0";
	
	/* Zeiger auf NULL */
	m_Port = NULL; 
	
	/* Diese Devices sind korrekt fuer Linux 2.4 TODO: devfs */
	m_defaultBtDevice="/dev/rfcomm0";
	m_defaultIrdaDevice="/dev/ircomm0";
	m_defaultSerialDevice="/dev/ttyS0";

	
	/* Lese die Konfiguration aus dem Configfile ein. */
	readSettings();
	
	/* Spitze die Config in das Settings Tab ab. */
	setSettingsView();
	
	/* Verbinde automatisch mit Telephon, wenn das Flag in der Config gesetzt ist.*/
	if (m_ConnectStartup == "Yes")
	{
		phoneConnect();
	}
	else
	{
		phoneGetInfo();
  	}
	
	/* Vestecke die ID Spalte der SMS Ansicht. BluePhone braucht diese Info, der Benutzer nicht. */
	//DOESNOTWORK
	//listViewSms->hideColumn(0);
	
	/* Vestecke die ID Spalte der Kontakte Ansicht. BluePhone braucht diese Info, der Benutzer nicht. */
	//DOESNOTWORK
	//listViewContacts->hideColumn(0);
	
}


/**
  * BluePhone Destruktor
  */
BluePhone::~BluePhone()
{
}



/**
  * Diese Methode liest, falls vorhanden, die aktuellen Settings aus dem Configfile ein. 
  */
void BluePhone::readSettings()
{
	/* Erstelle BlueSettings Objekt */
	BlueSettings mySettings;
	
	/* Setzen der Werte */
	m_settingsAll = mySettings.readSettings();
	
	m_Device = m_settingsAll.section(',',0,0);
	m_BtAddress = m_settingsAll.section(',',1,1);
	m_BtChannel = m_settingsAll.section(',',2,2);
	m_ConnectStartup = m_settingsAll.section(',',3,3);
}


/** 
  * Diese Methode fuellt die Settings Form mit aktuellen Werten ab.
  */
void BluePhone::setSettingsView()
{
	if (m_Device == m_defaultBtDevice)
	{
		radioButtonSettingsBluetooth->toggle();
	}
	else if (m_Device == m_defaultIrdaDevice)
	{
		radioButtonSettingsIrDA->toggle();
	}
	else if (m_Device == m_defaultSerialDevice)
	{
		radioButtonSettingsSerial->toggle();
	}
	
	if (m_ConnectStartup == "Yes")
	{
		radioButtonSettingsYes->toggle();
	}
	else if (m_ConnectStartup == "No")
	{
		radioButtonSettingsNo->toggle();
	}
	
	lineEditSettingsAddress->setText(m_BtAddress);
		
	m_BtChannelInt = m_BtChannel.toInt();
	spinBoxSettingsPort->setValue(m_BtChannelInt);
}


/** 
  * Diese Methode liest Informationen ueber eine File aus
  * und stellt diese im Tab "File" dar.
  * Parameter: QString "Pfad zur Datei"
  */
void BluePhone::showMetaInfo(const QString& pathToFile)
{
	/* Erstelle ein "KFileItem" Objekt */
	KFileItem myFileInfo(KFileItem::Unknown, KFileItem::Unknown, pathToFile, false);
	
	/* Ist die Datei lesbar? */
	if(myFileInfo.isReadable() && myFileInfo.isFile())
	{
		/* Ermittle den Mime Type der Datei */
		if(myFileInfo.isMimeTypeKnown())
		{
			lineEditFileType->setText(myFileInfo.mimeComment());
		}
		else
		{
			lineEditFileType->setText("Unknown");
		}

		/* Ermittle die Groesse der Datei */
		lineEditFileSize->setText(KIO::convertSize(myFileInfo.size()));

		/* Ermittle Metainformationen der Datei */
		const KFileMetaInfo info = myFileInfo.metaInfo();
		if(info.isValid())
		{
			const KFileMetaInfoItem sizeInfo = info.item(KFileMimeTypeInfo::Size);

			lineEditFileDimensions->setText(sizeInfo.string());
		}
		else
		{
			lineEditFileDimensions->setText("---");
		}

	}
	else
	{

		/* Loesche den Inhalt der Felder */
		lineEditFileType->clear();
		lineEditFileSize->clear();
		lineEditFileDimensions->clear();
	}
}


/**
  * Dieser Slot speichert die Settings.
  * Die Configuration wird in ~/.kde/share/config/bluephonerc gesichert
  */
void BluePhone::settingsSave()
{
	/* Frage die Werte aus der Maske ab */
	if (radioButtonSettingsBluetooth->isChecked())
	{
		m_settingsDevice = m_defaultBtDevice;
	}
	else if (radioButtonSettingsIrDA->isChecked())
	{
		m_settingsDevice = m_defaultIrdaDevice;
	}
	else if (radioButtonSettingsSerial->isChecked())
	{
		m_settingsDevice = m_defaultSerialDevice;
	}
	
	if (radioButtonSettingsYes->isChecked())
	{
		m_settingsConnectStartup = "Yes";
	}
	else if (radioButtonSettingsNo->isChecked())
	{
		m_settingsConnectStartup = "No";
	}
	
	m_settingsBtAddress = lineEditSettingsAddress->text();
	m_settingsBtChannel = spinBoxSettingsPort->value();
	
	
	/* Falls wir noch mit dem Telephon verbunden sind, nun trennen. */
	if (myLink.getState())
	{	
		phoneDisconnect();
	}

	/* Erstelle BlueSettings Objekt. */
	BlueSettings mySettings;
	
	/* Speichere die Settings. */		
	mySettings.writeSettings(m_settingsDevice, m_settingsBtAddress, m_settingsBtChannel, m_settingsConnectStartup);
	
	/* Lese die Settings gleich ein. */
	readSettings();
	
	QMessageBox::information( this, 
							tr("Information"),
							tr("Config saved.\nNow connect your phone." ) 
							);
}


/**
  * Dieser Slot zeigt eine Messagebox an, die erklaert, wie die Bluetooth Adresse des Telephons gefunden werden kann.
  */
void BluePhone::settingsGetAddress()
{
	QMessageBox::information( this, 
							tr("How to get the BTADR"),
							tr("How to get the bluetooth address of your phone\n\n\n" 
								"Open a Unix Shell and type in the following comand:\n\n" 
								"hcitool scan\n\n" 
								"The output of this comand will show you the bluetooth address and the name of your phone.\n"
								"A bluetooth address looks like this: 00:0A:D9:F7:3C:B6\n\n" 
								"You can use copy & paste to put this value in the \"Bluetooth Address\" field.") 
							);
}


/**
  * Dieser Slot erstellt ein neues SMS an den selektierten Kontakt.
  */
void BluePhone::contactsSms()
{
	if (myLink.getState())
	{
		/* liefert das selektierte Item aus listViewContacts zurueck */
		QListViewItem *selectedContact = listViewContacts->selectedItem();

		/* selectedItem liefert 0 zurueck, falls keine Selektion besteht */
		if (selectedContact != 0)
		{
			/* Inhalt von listViewContacts (die Nummer) in die private Variable abspeichern. */
			m_phoneSmsNumber = selectedContact->text(3);
			
			/* Erstelle einen NewSms Dialog, teile ihm mit, dass dies eine neue Nachricht mit definiertem 
			   Empfaenger ist (fromcontact) und uebergebe den Empfaenger als weiteren Parameter an den Konstruktor. */
			NewSms *myNewSmsDlg = new NewSms(this, 0, false, 0, "fromcontact", m_phoneSmsNumber);
		
		  	/* Erstelle eine Signal-Slot Verbindung 
				von: myNewSmsDlg::transmitSms (Quelle rsp. Signal)  
		   		zu:  this::sendSmsToPhone (this = BluePhone, Senke rsp. Slot) 
		   		Es werden die folgenden Parameter uebermittelt: QString Receiver, QString Message, int index, int id */
			QObject::connect(
						myNewSmsDlg, SIGNAL(transmitSms(const QString&, const QString&, int, int)),
						this, SLOT(sendSmsToPhone(const QString&, const QString&, int, int))
						);
								
			/* Erstelle eine Signal-Slot Verbindung fuer die gruene LED Status Anzeige. */
			QObject::connect(
						myNewSmsDlg, SIGNAL(transmitLedStateGreen()),
						this, SLOT(setLedGreen())
						);
			
			/* Setzte die gelbe LED. Dies muss leider hier geschehen, ein Aufruf vom "Send" Button des myNewSmsDlg
	   	   	   her kommt zu spaet, da der Fensterinhalt des Hauptfensters wird erst nach schliessen des Dialoges 
		  	   neu gezeichnet wird. */			
			setLedOrange();
						
			/* Zeige den NewSms Dialog an */
			myNewSmsDlg->show();
		}
		else
		{
  			QMessageBox::information( this, 
							tr("Information"),
							tr("Please select a Contact")
							);
		}
		
		setLedGreen();					
	}
	else
	{
  		QMessageBox::warning( this, 
						tr("Warning"),
						tr("Phone is not connected")
						);
	}
}


/** 
  * Dieser Slot ruft den selektierten Kontakt an. 
  */
void BluePhone::contactsCall()
{
	if (myLink.getState())
	{
		/* liefert das selektierte Item aus listViewContacts zurueck */
		QListViewItem *selectedContact = listViewContacts->selectedItem();
		
		/* selectedItem liefert 0 zurueck, falls keine Selektion besteht */
		if (selectedContact != 0)
		{
			/* setze das AT Kommando fuer Call zuasammen */
			m_AtCommand = "at*evd=\"";
			m_AtCommand.append(selectedContact->text(3));
			m_AtCommand.append("\"\r");

			myLink.talktoPhone(m_Port, m_AtCommand);
		}
		else
		{
  			QMessageBox::information( this, 
							tr("Information"),
							tr("Please select a contact")
							);
		}
		
		setLedGreen();
  	}
  	else
	{
  		QMessageBox::warning( this, 
							tr("Warning"),
							tr("Phone is not connected")
							);
	}
}


/** 
  * Dieser Slot ueberschreibt das Telephonbuch auf dem Telephon mit einem Backup.  
  */
void BluePhone::contactsRestore()
{
	if (myLink.getState())
	{
		/* "Leere listViewContacts */
		listViewContacts->clear();
		
		/* "Open File" Dialog */
		m_FileName = QFileDialog::getOpenFileName(
							tr("/home"),
							tr("Text files (*.txt);;All files (*.*)"),
    	          	     	this,
    	               		tr("Open File"),
							tr("Open Backup")
					   		);
		
		/*  User abfragen, ob er wirklich restoren will */
		bool MessageboxAnswer = QMessageBox::question( this, 
										tr("Question"),
										tr("Do you really want to restore your phonebook from %1 ?\n"
											"This will overwrite all contact entries on your phone!")
										.arg( m_FileName ),
           								tr("&No"), tr("&Yes"),
           								QString::null, 1, 0 );
					   		
		/* Hat der User ein File gewaehlt oder Cancel gedrueckt und will er wirklich? */
		if (m_FileName && MessageboxAnswer)
		{
			/* was aus der guten, alten 3. Semester Zeit. Oeffne einen ifstream */ 
			std::ifstream inputFile(m_FileName.latin1());
			
			/* 255 Bytes grossen Char Array erstellen. Brauchen wir fuer getline. */
			char buffer[255];
			
			/* lese das Backup File bis zum EOF (EndOfFile) ein und spitze die Eintraege in listViewContacts ab */ 		   
			while (!inputFile.eof())
    		{
				/* Hole eine Zeile */
				inputFile.getline(buffer, 255);
				m_phoneContactEntry.setLatin1(buffer);			   
								
				/* Ermittle die einzeln Werte des Eintrages*/
				m_phoneContactId = m_phoneContactEntry.section(':',0,0);
				
				m_phoneContactType = m_phoneContactEntry.section(':',1,1);
				if ( m_phoneContactType == "Home" )
				{ m_phoneContactType = "H"; }
				else if ( m_phoneContactType == "Work" )
				{ m_phoneContactType = "W"; }
				else if ( m_phoneContactType == "Mobile" )
				{ m_phoneContactType = "M"; }
				else if ( m_phoneContactType == "Fax" )
				{ m_phoneContactType = "F"; }
				else 
				{ m_phoneContactType = "O"; }
				
				m_phoneContactName = m_phoneContactEntry.section(':',2,2);
				m_phoneContactNumber = m_phoneContactEntry.section(':',3,3);
				
				/* Der Zeichensatz von m_phoneContactName muss noch fuer das Telephon umgewandelt werden*/
				m_phoneContactName = myConvert.convert_to_gsm(m_phoneContactName);
				
				/* setze das AT Kommando fuer das Telephon zusammen. Siehe AT Referenz. */
				m_AtCommand = "at+cpbw=";
				m_AtCommand.append(m_phoneContactId);
				m_AtCommand.append(",\"");
				m_AtCommand.append(m_phoneContactNumber);
				m_AtCommand.append("\",,\"");
				m_AtCommand.append(m_phoneContactName);
				m_AtCommand.append("/");
				m_AtCommand.append(m_phoneContactType);
				m_AtCommand.append("\"\r");
	
				/* uebermittle dem Telephon den Telephonbuch Eintrag */
				myLink.talktoPhone(m_Port, m_AtCommand);
					
				/* Sortiere die Eintraege dem Namen nach */
        		listViewContacts->setSorting(2);
				listViewContacts->sort();
			}
			
			/* Lese das Telephonbuch vom Telephon ein. So kann der User ueberpruefen, ob alles geklappt hat */
			contactsRead();
			
			QMessageBox::information( this, 
								tr("Information"),
								tr("Phonebook from %1 restored" )
           		     			.arg( m_FileName ) 
								);
		}
		
		setLedGreen();
  	}
  	else
	{
  		QMessageBox::warning( this, 
							tr("Warning"),
							tr("Phone is not connected")
							);
	}
}


/** 
  * Dieser Slot erstellt ein Textfile Backup aus allen Kontakten. 
  */
void BluePhone::contactsBackup()
{
	if (myLink.getState())
	{
		/* nur sichern, wenn das Telephonbuch bereits gelesen wurde */
		if (listViewContacts->childCount() != 0)
		{
			/* "Save File" Dialog */
			m_FileName = QFileDialog::getSaveFileName(
     	               tr("/home"),
    	               tr("Text files (*.txt);;All files (*.*)"),
    	               this,
    	               tr("Save File"),
     	               tr("Save Phonebook Backup As")
					   );
			
			/* liefert das erste Item aus listViewContacts zurueck */
			QListViewItem *Contact = listViewContacts->firstChild();
			
			/* Hat der User ein File gewaehlt oder Cancel gedrueckt? */
			if (m_FileName)
			{
				/* was aus der guten, alten 3. Semester Zeit. Oeffne einen ofstream */ 
				std::ofstream outputFile(m_FileName.latin1());
				
				/* Spitze listViewContacts in das outputFile ab */
				while (Contact)
				{
					outputFile << Contact->text(0).latin1() << ":";
					outputFile << Contact->text(1).latin1() << ":";
					outputFile << Contact->text(2).latin1() << ":";
					outputFile << Contact->text(3).latin1() << "\n";
					
					Contact = Contact->nextSibling();
				}
				
				QMessageBox::information( this, 
								tr("Information"),
								tr("Phonebook as %1 saved" )
           		     			.arg( m_FileName ) 
								);				
			}		
		}
		else
		{
  			QMessageBox::information( this, 
							tr("Information"),
							tr("Please read the phonebook first")
							);
		}
		
		setLedGreen();									
	}
	else
	{
  		QMessageBox::warning( this, 
						tr("Warning"),
						tr("Phone is not connected")
						);
	}
}


/** 
  * Dieser Slot loescht den selektierten Kontakt. 
  */
void BluePhone::contactsDelete()
{
	if (myLink.getState())
	{
		/* liefert das selektierte Item aus listViewContacts zurueck */
		QListViewItem *selectedContact = listViewContacts->selectedItem();

		/* selectedItem liefert 0 zurueck, falls keine Selektion besteht */
		if (selectedContact != 0)
		{
			/* Inhalt von selectedContact in die privaten Variablen abspeichern. */
			m_phoneContactId = selectedContact->text(0);
			m_phoneContactName = selectedContact->text(2);
			
			/*  User abfragen, ob er wirklich loeschen will */
			bool MessageboxAnswer = QMessageBox::question( this, 
										tr("Question"),
										tr("Do you really want to delete contact %1 ?")
										.arg( m_phoneContactName ),
           								tr("&No"), tr("&Yes"),
           								QString::null, 1, 0 );
			
			if (MessageboxAnswer)
			{
				/* setze das AT Kommando fuer das Telephon zusammen. Siehe AT Referenz. */
				m_AtCommand = "at+cpbw=";
				m_AtCommand.append(m_phoneContactId);
				m_AtCommand.append("\r");
				
				/* uebermittle dem Telephon den zu leoschenden Eintrag */
				myLink.talktoPhone(m_Port, m_AtCommand);

				/* Lese das Telephonbuch vom Telephon ein. So kann der User ueberpruefen, on alles geklappt hat */
				contactsRead();
			}
			
			setLedGreen();
			
		}
		else
		{
  			QMessageBox::information( this, 
							tr("Information"),
							tr("Please select a contact")
							);
		}
		
		setLedGreen();				
	}
	else
	{
  		QMessageBox::warning( this, 
						tr("Warning"),
						tr("Phone is not connected")
						);
	}
}


/** 
  * Dieser Slot editiert den selektierten Kontakt. 
  */
void BluePhone::contactsEdit()
{
	if (myLink.getState())
	{
		/* liefert das selektierte Item aus listViewContacts zurueck */
		QListViewItem *selectedContact = listViewContacts->selectedItem();

		/* selectedItem liefert 0 zurueck, falls keine Selektion besteht */
		if (selectedContact != 0)
		{
			/* Inhalt von selectedContact in die privaten Variablen abspeichern. */
			m_phoneContactId = selectedContact->text(0);
			m_phoneContactType = selectedContact->text(1);
			m_phoneContactName = selectedContact->text(2);
			m_phoneContactNumber = selectedContact->text(3);

			/* Erstelle einen EditContact Dialog, 
			 * Ich verwende fuer Edit den selben Dialog wie fuer New, hier editiere ich, also geht als Parameter 5 "edit" mit
			 * Weiter braucht ein Edit Dialog braucht die Daten des selektierten Eintrages, Parameter 6 - 9
			 */
			NewContact *myEditContactDlg = new NewContact(this, 0, false, 0, "edit", m_phoneContactId, m_phoneContactType, m_phoneContactName, m_phoneContactNumber);
		
			/* Erstelle eine Signal-Slot Verbindung 
				von: myEditContactDlg::transmitContact (Quelle rsp. Signal)  
		   		zu:  this::sendContactToPhone (this = BluePhone, Senke rsp. Slot) 
				Es werden die folgenden Parameter uebermittelt: ContactId, ContactType, ContactName, ContactNumber*/
			QObject::connect(
						myEditContactDlg, SIGNAL(transmitContact(const QString&, const QString&, const QString&, const QString&)),
						this, SLOT(sendContactToPhone(const QString&, const QString&, const QString&, const QString&))
						);

			/* Erstelle eine Signal-Slot Verbindung fuer die gruene LED Status Anzeige. */
			QObject::connect(
						myEditContactDlg, SIGNAL(transmitLedStateGreen()),
						this, SLOT(setLedGreen())
						);

			/* Setzte die gelbe LED. Dies muss leider hier geschehen, ein Aufruf vom "Send" Button des myEditContactDlg
		   	   her kommt zu spaet, da der Fensterinhalt des Hauptfensters wird erst nach schliessen des Dialoges 
			   neu gezeichnet wird. */			
			setLedOrange();
												
			/* Zeige den NewContact Dialog an */
			myEditContactDlg->show();
		}
		else
		{
  			QMessageBox::information( this, 
							tr("Information"),
							tr("Please select a contact")
							);
		}
		
		setLedGreen();				
	}
	else
	{
  		QMessageBox::warning( this, 
							tr("Warning"),
							tr("Phone is not connected")
							);
	}
}


/**
  * Dieser Slot erstellt einen neuen Kontakt.
  */
void BluePhone::contactsNew()
{
	if (myLink.getState())
	{
		/* Erstelle einen NewContact Dialog. Dies ist der default, deshalb keine weiteren Parameter an den Konstruktor. */
		NewContact *myNewContactDlg = new NewContact(this);
		
		/* Erstelle eine Signal-Slot Verbindung 
			von: myNewContactDlg::transmitContact (Quelle rsp. Signal)  
		   zu:  this::createNewContact (this = BluePhone, Senke rsp. Slot) 
		   Es werden die folgenden Parameter uebermittelt: ContactId, ContactType, ContactName, ContactNumber*/
		QObject::connect(
					myNewContactDlg, SIGNAL(transmitContact(const QString&, const QString&, const QString&, const QString&)),
					this, SLOT(sendContactToPhone(const QString&, const QString&, const QString&, const QString&))
					);

		/* Erstelle eine Signal-Slot Verbindung fuer die gruene LED Status Anzeige. */
		QObject::connect(
					myNewContactDlg, SIGNAL(transmitLedStateGreen()),
					this, SLOT(setLedGreen())						);

		/* Setzte die gelbe LED. Dies muss leider hier geschehen, ein Aufruf vom "Send" Button des myNewContactDlg
	   	   her kommt zu spaet, da der Fensterinhalt des Hauptfensters wird erst nach schliessen des Dialoges 
		   neu gezeichnet wird. */			
		setLedOrange();
							
		/* Zeige den NewContact Dialog an */
		myNewContactDlg->show();
		
		
	}
	else
	{
  		QMessageBox::warning( this, 
							tr("Warning"),
							tr("Phone is not connected")
							);
	}
}


/**
  * Dieser Slot liest das Telephonbuch des Telephons ein.
  */
void BluePhone::contactsRead()
{
	if (myLink.getState())
	{	
		listViewContacts->clear();	
	
    	/* teile dem Telephon mit, dass wir das Telephonbuch des Flash Speichers benutzen wollen. (ME)
		   weitere waeren: SM: Simcard, BC: Business Card, usw. Siehe AT-Ref Seite 102 */
		myLink.talktoPhone(m_Port, "at+cpbs=\"ME\"\r");
  
		/* lese die maximale Anzahl Telephonbuch Entraege aus */
		m_phoneAnswer = myLink.talktoPhone(m_Port, "at+cpbr=?\r");
		m_phoneContactCount = m_phoneAnswer.join(":");
		m_phoneContactCount = m_phoneContactCount.section(':',2,2);
		m_phoneContactCount = m_phoneContactCount.section(')',0,0);
		m_phoneContactCount = m_phoneContactCount.section('(',1,1);
		m_phoneContactCount = m_phoneContactCount.section('-',1,1);
			   
		/* lese das gesamte Telephonbuch aus. eine Abfrage nur der belegten Plaetze ist nicht moeglich*/
		m_phoneAnswer = myLink.talktoPhone(m_Port, "at+cpbr=1,"+m_phoneContactCount+"\r");
		
		/* spitze die Eintraege  mit Hilfe des QStringList Iterators in listViewContacts ab. Siehe QT Doku zu QStringList.*/
		for (QStringList::ConstIterator it = m_phoneAnswer.begin(); it != m_phoneAnswer.end(); ++it)
		{
			/* Zeichensatz konvertieren. Siehe BlueConvert */
			m_phoneContactEntry.setLatin1(*it);
			m_phoneContactEntry = myConvert.convert_from_gsm(m_phoneContactEntry);
	  
			/* Status Meldungen wie OK und so sollen nicht abgespitzt werden. Alle "richtigen" Eintraege beginnen mit +CPBR */
			if (m_phoneContactEntry.contains("+CPBR:")) 
			{
				/* Ermittle die einzeln Werte des Eintrages*/
				m_phoneContactEntry = m_phoneContactEntry.section(':',1,1);
				
				m_phoneContactId = m_phoneContactEntry.section(',',0,0);
				m_phoneContactId = m_phoneContactId.stripWhiteSpace();
				
				m_phoneContactType = m_phoneContactEntry.section(',',3,3);
				m_phoneContactType = m_phoneContactType.section('/',1,1);
				m_phoneContactType = m_phoneContactType.left(1);
				
				if ( m_phoneContactType == "H" )
				{ m_phoneContactType = "Home"; }
				else if ( m_phoneContactType == "W" )
				{ m_phoneContactType = "Work"; }
				else if ( m_phoneContactType == "M" )
				{ m_phoneContactType = "Mobile"; }
				else if ( m_phoneContactType == "F" )
				{ m_phoneContactType = "Fax"; }
				else 
				{ m_phoneContactType = "Other"; }
				
				m_phoneContactName = m_phoneContactEntry.section(',',3,3);
				m_phoneContactName = m_phoneContactName.section('/',0,0);
				m_phoneContactName = m_phoneContactName.section('"',1,1);
				
				m_phoneContactNumber = m_phoneContactEntry.section(',',1,1);
				m_phoneContactNumber = m_phoneContactNumber.section('"',1,1);
				
				
				/* Erstelle mit den gewonnenen Werten ein neues QListViewItem in listViewContacts  */
				new QListViewItem(listViewContacts, m_phoneContactId, m_phoneContactType, m_phoneContactName, m_phoneContactNumber);
			}
		}
		
		/* Sortiere die Eintraege dem Namen nach */
   		listViewContacts->setSorting(2);
		listViewContacts->sort();
				
		/* Vestecke die ID Spalte der Kontakte Ansicht. BluePhone braucht diese Info, der Benutzer nicht. */
		//DOESNOTWORK
		//listViewContacts->hideColumn(0);
		
		setLedGreen();
	}
	else
	{
  		QMessageBox::warning( this, 
							tr("Warning"),
							tr("Phone is not connected")
							);
	}
}


/**
  * Dieser Slot erstellt ein neues SMS.
  */
void BluePhone::smsNew()
{
	if (myLink.getState())
	{
		/* Erstelle einen NewSms Dialog. Dies ist der default, deshalb keine weiteren Parameter an den Konstruktor. */
		NewSms *myNewSmsDlg = new NewSms(this);
		
		/* Erstelle eine Signal-Slot Verbindung 
			von: myNewSmsDlg::transmitSms (Quelle rsp. Signal)  
		   zu:  this::sendSmsToPhone (this = BluePhone, Senke rsp. Slot) 
		   Es werden die folgenden Parameter uebermittelt: QString Receiver, QString Message, int index, int id */
		QObject::connect(
					myNewSmsDlg, SIGNAL(transmitSms(const QString&, const QString&, int, int)),
					this, SLOT(sendSmsToPhone(const QString&, const QString&, int, int))
					);
							
			/* Erstelle eine Signal-Slot Verbindung fuer die gruene LED Status Anzeige. */
			QObject::connect(
						myNewSmsDlg, SIGNAL(transmitLedStateGreen()),
						this, SLOT(setLedGreen())
						);

		/* Setzte die gelbe LED. Dies muss leider hier geschehen, ein Aufruf vom "Send" Button des myNewSmsDlg
	   	   her kommt zu spaet, da der Fensterinhalt des Hauptfensters wird erst nach schliessen des Dialoges 
		   neu gezeichnet wird. */			
		setLedOrange();
													
		/* Zeige den NewSms Dialog an */
		myNewSmsDlg->show();
	}
	else
	{
  		QMessageBox::warning( this, 
							tr("Warning"),
							tr("Phone is not connected")
							);
	}
}


/**
  * Dieser Slot loescht alle SMS auf dem Telephon.
  */
void BluePhone::smsDeleteAll()
{
	if (myLink.getState())
	{
		/* nur loeschen, wenn die SMS Liste bereits gelesen wurde */
		if (listViewSms->childCount() != 0)
		{
			/*  User abfragen, ob er wirklich alle SMS loeschen will */
			bool MessageboxAnswer = QMessageBox::question( this, 
										tr("Question"),
										tr("Do you really want to delete all SMS on your phone?"),
           								tr("&No"), tr("&Yes"),
           								QString::null, 1, 0 );
			
			if (MessageboxAnswer)
			{
				/* teile dem Telephon mit, dass wir die SMS Daten des Flash Speichers benutzen wollen. (ME)
					weitere waeren: SM: Simcard, TL: Templeate. */
				myLink.talktoPhone(m_Port, "at+cpms=\"ME\"\r");
				
				/* lese den ersten Eintrag aus der SMS Liste. */
				QListViewItem *Sms = listViewSms->firstChild();

        		/* loesche die SMS der Reihe nach. */
				while(Sms)
        		{
     				/* setze das AT Kommando fuer das Telephon zusammen. Siehe AT Referenz. */
					m_AtCommand = "at+cmgd=";
					m_AtCommand.append(Sms->text(0));
					m_AtCommand.append("\r");
								
					/* uebermittle dem Telephon den zu leoschenden Eintrag */
					myLink.talktoPhone(m_Port, m_AtCommand);
					
					/* Gehe zum naechsten SMS. */
         			Sms = Sms->nextSibling();
        		}		

				/* Lese die SMS Liste vom Telephon ein. So kann der User ueberpruefen, on alles geklappt hat */
				smsRead();
			}
		}
		else
		{
  			QMessageBox::information( this, 
							tr("Information"),
							tr("Please read the SMS list first")
							);
		}
		
		setLedGreen();									
	}
	else
	{
  		QMessageBox::warning( this, 
						tr("Warning"),
						tr("Phone is not connected")
						);
	}
}


/**
  * Dieser Slot loescht ein SMS.
  */
void BluePhone::smsDelete()
{
	if (myLink.getState())
	{
		/* liefert das selektierte Item aus listViewSms zurueck */
		QListViewItem *selectedSms = listViewSms->selectedItem();

		/* selectedItem liefert 0 zurueck, falls keine Selektion besteht */
		if (selectedSms != 0)
		{
			/* Inhalt von listViewSms in die privaten Variablen abspeichern. */
			m_phoneSmsId = selectedSms->text(0);
						
			/*  User abfragen, ob er wirklich loeschen will */
			bool MessageboxAnswer = QMessageBox::question( this, 
										tr("Question"),
										tr("Do you really want to delete SMS with ID %1 ?")
										.arg( m_phoneSmsId ),
           								tr("&No"), tr("&Yes"),
           								QString::null, 1, 0 );
			
			if (MessageboxAnswer)
			{
				/* teile dem Telephon mit, dass wir die SMS Daten des Flash Speichers benutzen wollen. (ME)
					weitere waeren: SM: Simcard, TL: Templeate. */
				myLink.talktoPhone(m_Port, "at+cpms=\"ME\"\r");
							
				/* setze das AT Kommando fuer das Telephon zusammen. Siehe AT Referenz. */
				m_AtCommand = "at+cmgd=";
				m_AtCommand.append(m_phoneSmsId);
				m_AtCommand.append("\r");
								
				/* uebermittle dem Telephon den zu leoschenden Eintrag */
				myLink.talktoPhone(m_Port, m_AtCommand);

				/* Lese die SMS Liste vom Telephon ein. So kann der User ueberpruefen, on alles geklappt hat */
				smsRead();
			}
		}
		else
		{
  			QMessageBox::information( this, 
							tr("Information"),
							tr("Please select a SMS")
							);
		}
		
		setLedGreen();				
	}
	else
	{
  		QMessageBox::warning( this, 
						tr("Warning"),
						tr("Phone is not connected")
						);
	}
}


/**
  * Dieser Slot speichert alle Kurznachrichten in einem Textfile ab.
  */
void BluePhone::smsSaveAll()
{
	if (myLink.getState())
	{
		/* nur sichern, wenn die SMS Liste bereits gelesen wurde */
		if (listViewSms->childCount() != 0)
		{
			/* "Save File" Dialog */
			m_FileName = QFileDialog::getSaveFileName(
     	               tr("/home"),
    	               tr("Text files (*.txt);;All files (*.*)"),
    	               this,
    	               tr("Save File"),
     	               tr("Save SMS Backup As")
					   );
			
			/* liefert das erste Item aus listViewSms zurueck */
			QListViewItem *Sms = listViewSms->firstChild();
			
			/* Hat der User ein File gewaehlt oder Cancel gedrueckt? */
			if (m_FileName)
			{
				/* was aus der guten, alten 3. Semester Zeit. Oeffne einen ofstream */ 
				std::ofstream outputFile(m_FileName.latin1());
				
				/* Spitze listViewSms in das outputFile ab */
				while (Sms)
				{
					outputFile << "From: " << Sms->text(3).latin1() << "   ";
					outputFile << "Date: " << Sms->text(2).latin1() << "   ";
					outputFile << "State: " << Sms->text(1).latin1() << " \n";
					outputFile << "Message: " << Sms->text(4).latin1() << " \n\n\n";
					
					Sms = Sms->nextSibling();
				}
				
				QMessageBox::information( this, 
								tr("Information"),
								tr("List of all SMS as %1 saved" )
           		     			.arg( m_FileName ) 
								);				
			}		
		}
		else
		{
  			QMessageBox::information( this, 
							tr("Information"),
							tr("Please read the SMS list first")
							);
		}									
	}
	else
	{
  		QMessageBox::warning( this, 
						tr("Warning"),
						tr("Phone is not connected")
						);
	}
}


/**
  * Dieser Slot speichert eine Kurznachricht in einem Textfile ab.
  */
void BluePhone::smsSave()
{
	if (myLink.getState())
	{
		/* liefert das selektierte Item aus listViewSms zurueck */
		QListViewItem *selectedSms = listViewSms->selectedItem();

		/* selectedItem liefert 0 zurueck, falls keine Selektion besteht */
		if (selectedSms != 0)
		{
			/* "Save File" Dialog */
			m_FileName = QFileDialog::getSaveFileName(
     	               tr("/home"),
    	               tr("Text files (*.txt);;All files (*.*)"),
    	               this,
    	               tr("Save File"),
     	               tr("Save SMS Backup As")
					   );
						
			/* Hat der User ein File gewaehlt oder Cancel gedrueckt? */
			if (m_FileName)
			{
				/* was aus der guten, alten 3. Semester Zeit. Oeffne einen ofstream */ 
				std::ofstream outputFile(m_FileName.latin1());
				
				/* Spitze das selektierte SMS in das outputFile ab */
				outputFile << "From: " << selectedSms->text(3).latin1() << "   ";
				outputFile << "Date: " << selectedSms->text(2).latin1() << "   ";
				outputFile << "State: " << selectedSms->text(1).latin1() << " \n";
				outputFile << "Message: " << selectedSms->text(4).latin1() << " \n";
								
				QMessageBox::information( this, 
								tr("Information"),
								tr("SMS as %1 saved" )
           		     			.arg( m_FileName ) 
								);				
			}		
		}
		else
		{
  			QMessageBox::information( this, 
							tr("Information"),
							tr("Please select a SMS first")
							);
		}									
	}
	else
	{
  		QMessageBox::warning( this, 
						tr("Warning"),
						tr("Phone is not connected")
						);
	}
}


/**
  * Dieser Slot liest die Kurznachrichten vom Telephon ein.
  */
void BluePhone::smsRead()
{
	if (myLink.getState())
	{	
		listViewSms->clear();	

  	 	/* teile dem Telephon mit, dass wir die SMS Daten des Flash Speichers benutzen wollen. (ME)
		   weitere waeren: SM: Simcard, TL: Templeate. */
		myLink.talktoPhone(m_Port, "at+cpms=\"ME\"\r");

		/* lese alle SMS aus dem Telephon aus. Der Parameter bestimmt:
		   0 = empfangene, ungelesene Nachrichten, 
		   1 = empfangene, gelesene Nachrichten,
		   4 = alle Nachrichten,
		   16 = template  Nachrichten */
		m_phoneAnswer = myLink.talktoPhone(m_Port, "at+cmgl=4\r");
						
		/* spitze die Eintraege  mit Hilfe des QStringList Iterators in listViewSms ab. Siehe QT Doku zu QStringList.*/
		for (QStringList::ConstIterator it = m_phoneAnswer.begin(); it != m_phoneAnswer.end(); ++it)
		{
			/* Zeichensatz konvertieren. */
			m_phoneSmsEntry.setLatin1(*it);
			
			/* Status Meldungen wie OK und so sollen nicht abgespitzt werden. Alle "richtigen" Eintraege beginnen mit +CMGL */
			if (m_phoneSmsEntry.contains("+CMGL:")) 
    		{	
				/* Ermittle die einzeln Werte des Eintrages*/
				m_phoneSmsId = m_phoneSmsEntry.section(':',1,1);
				m_phoneSmsId = m_phoneSmsId.section(',',0,0);
				m_phoneSmsId = m_phoneSmsId.stripWhiteSpace();
				
    			m_phoneSmsStatus = m_phoneSmsEntry.section(',',1,1);
	
				if ( m_phoneSmsStatus == "3" )
					{ m_phoneSmsStatus = "Sent"; }
				else if ( m_phoneSmsStatus == "1" )
					{ m_phoneSmsStatus = "Read"; }
				else if ( m_phoneSmsStatus == "0" )
					{ m_phoneSmsStatus = "Unread"; }
				else
					{ m_phoneSmsStatus = "Unknown"; }
 					
				/* Eine SMS besteht aus zwei Teilen. Der erste enthaelt die ID und den Status, 
				   der zweite die Nummer und die Message in einem "IRA-character long hexadecimal" Format
				   (Terminologie Ericsson). Nun wollen wir diese Daten, deshalb wechseln wir zum naechsten Eintrag */
      			it++;
				
				/* erneut Zeichensatz konvertieren. Irgendwie hat mir dieses Telephon zu viele Datenformate... */
      			m_phoneSmsEntry.setLatin1(*it);
      								
				/* Erstelle ein neues BlueConvert Objekt */
				BlueConvert *smsData = new BlueConvert();
	  
	  			/* Uebersetze das 7Bit Hex Format(PDU) in einen nomalen uuencode String */
      			smsData->extractPduData(m_phoneSmsEntry);
	  
				/* Hole die gewonnenen Daten ab. */
				m_phoneSmsNumber = smsData->getSender();
				m_phoneSmsMessage = smsData->getMessage();
				m_phoneSmsDateTmp = smsData->getDate();
				
				/* Formatiere das Datum etwas leserlicher. Gesendete SMS haben kein Datum. */
				if ( m_phoneSmsStatus == "Sent" )
				{
					m_phoneSmsDate = " n.a.";
				}
				else
				{
					m_phoneSmsDate = m_phoneSmsDateTmp.mid(0,2);
					m_phoneSmsDate.append("-");
					m_phoneSmsDate.append(m_phoneSmsDateTmp.mid(2,2));
					m_phoneSmsDate.append("-");
					m_phoneSmsDate.append(m_phoneSmsDateTmp.mid(4,2));
					m_phoneSmsDate.append(" ");
					m_phoneSmsDate.append(m_phoneSmsDateTmp.mid(6,2));
					m_phoneSmsDate.append(":");
					m_phoneSmsDate.append(m_phoneSmsDateTmp.mid(8,2));
				}
					
				/* Erstelle mit den gewonnenen Werten ein neues QListViewItem in listViewContacts  */
      			new QListViewItem(listViewSms, m_phoneSmsId, m_phoneSmsStatus, m_phoneSmsDate, m_phoneSmsNumber, m_phoneSmsMessage);
      
				
				/* Zu new gehoert delete. Ich will ja kein mega Speicher fressendes mini Programm schreiben. */
				delete smsData;
    		}
  		}
						
		/* Sortiere die Eintraege absteigend dem Datum nach. */
   		listViewSms->setSorting(2, false);
		listViewSms->sort();
			
		/* Vestecke die ID Spalte der SMS Ansicht. BluePhone braucht diese Info, der Benutzer nicht. */
		//DOESNOTWORK
		//listViewSms->hideColumn(0);
		
		setLedGreen();
	}
	else
	{
  		QMessageBox::warning( this, 
							tr("Warning"),
							tr("Phone is not connected")
							);
	}
}


/**
  * Dieser Slot zeigt eine Nachricht in einer Messagebox an.
  */
void BluePhone::smsShow()
{
	if (myLink.getState())
	{
		/* liefert das selektierte Item aus listViewSms zurueck */
		QListViewItem *selectedSms = listViewSms->selectedItem();

		/* selectedItem liefert 0 zurueck, falls keine Selektion besteht */
		if (selectedSms != 0)
		{
			/* Inhalt von listViewSms in die privaten Variablen abspeichern. */
			m_phoneSmsId = selectedSms->text(0);
			m_phoneSmsStatus = selectedSms->text(1);
			m_phoneSmsDate = selectedSms->text(2);
			m_phoneSmsNumber = selectedSms->text(3);
			m_phoneSmsMessage = selectedSms->text(4);
			
			m_phoneSmsToShow = "Message from: ";
			m_phoneSmsToShow.append(m_phoneSmsNumber);
			m_phoneSmsToShow.append("\nReceived at: ");
			m_phoneSmsToShow.append(m_phoneSmsDate);
			m_phoneSmsToShow.append("\n\nMessage: ");
			m_phoneSmsToShow.append(m_phoneSmsMessage);
			m_phoneSmsToShow.append("\n\nReply?");
			
			bool MessageboxAnswer = QMessageBox::information( this, 
										tr("View Message"),
										tr("%1")
										.arg( m_phoneSmsToShow ),
           								tr("&Yes"), tr("&No"),
           								QString::null, 1, 0 );
			
			if (!MessageboxAnswer)
			{							
				/* Zeige den ReplySms Dialog an */
				smsReply();
			}
		}
		else
		{
  			QMessageBox::information( this, 
							tr("Information"),
							tr("Please select a Message")
							);
		}					
	}
	else
	{
  		QMessageBox::warning( this, 
						tr("Warning"),
						tr("Phone is not connected")
						);
	}
}


/**
  * Dieser Slot bietet die Moeglichkeit, eine Nachricht zu beantworteten.
  */
void BluePhone::smsReply()
{
	if (myLink.getState())
	{
		/* liefert das selektierte Item aus listViewSms zurueck */
		QListViewItem *selectedSms = listViewSms->selectedItem();

		/* selectedItem liefert 0 zurueck, falls keine Selektion besteht */
		if (selectedSms != 0)
		{
			/* Inhalt von listViewSms in die privaten Variablen abspeichern. */
			m_phoneSmsNumber = selectedSms->text(3);
			m_phoneSmsMessage = selectedSms->text(4);
			
			/* Erstelle einen NewSms Dialog, teile ihm mit, dass dies eine Antwort ist (reply) und 
			   uebergebe den Sender und die Original Message als weitere Parameter an den Konstruktor. */
			NewSms *myNewSmsDlg = new NewSms(this, 0, false, 0, "reply", m_phoneSmsNumber, m_phoneSmsMessage);
		
		  	/* Erstelle eine Signal-Slot Verbindung 
				von: myNewSmsDlg::transmitSms (Quelle rsp. Signal)  
		   		zu:  this::sendSmsToPhone (this = BluePhone, Senke rsp. Slot) 
		   		Es werden die folgenden Parameter uebermittelt: QString Receiver, QString Message, int index, int id */
			QObject::connect(
						myNewSmsDlg, SIGNAL(transmitSms(const QString&, const QString&, int, int)),
						this, SLOT(sendSmsToPhone(const QString&, const QString&, int, int))
						);
					
			/* Erstelle eine Signal-Slot Verbindung fuer die gruene LED Status Anzeige. */
			QObject::connect(
						myNewSmsDlg, SIGNAL(transmitLedStateGreen()),
						this, SLOT(setLedGreen())
						);

			/* Setzte die gelbe LED. Dies muss leider hier geschehen, ein Aufruf vom "Send" Button des myNewSmsDlg
		   	   her kommt zu spaet, da der Fensterinhalt des Hauptfensters wird erst nach schliessen des Dialoges 
			   neu gezeichnet wird. */			
			setLedOrange();
																	
			/* Zeige den NewSms Dialog an */
			myNewSmsDlg->show();
		}
		else
		{
  			QMessageBox::information( this, 
							tr("Information"),
							tr("Please select a Message")
							);
		}					
	}
	else
	{
  		QMessageBox::warning( this, 
						tr("Warning"),
						tr("Phone is not connected")
						);
	}
}


/**
  * Dieser Slot sendet, ueber Obex, eine Datei an das Telphon. 
  */
void BluePhone::fileSend()
{
	if (myLink.getState())
	{
		struct BlueObex session;
	  	obex_t *handle;
  
	  	session.client_done = false;
	  	session.filename = m_UploadFilePath;

		/* Initialisiere OBEX */
	  	if (m_Device.contains("ircomm"))
		{
    		handle = obexInit(&session, 1, "", 0);
		}
  		else if (m_Device.contains("rfcomm"))
  		{
		    handle = obexInit(&session, 2, m_BtAddress, m_BtChannelInt);
	  	}
	  	else
	  	{
 			QMessageBox::critical( this, 
							tr("Error"),
							tr("Could not establish OBEX connection to %1" )
                			.arg( m_Device ) 
							);
	    	return;
  		}
	
  		if (!handle)
  		{
			QMessageBox::warning( this, 
							tr("Warning"),
							tr("Invalid OBEX handle")
							);
	    	return;
  		}
	
		/* Stelle die Obex Verbindung her */
	  	if (obexConnect(handle))
	  	{
			QMessageBox::warning( this, 
							tr("Warning"),
							tr("OBEX connection error")
							);
	    	return;
  		}
		
	  	/* Lade die Datei auf das Telephon */
	  	obexPutfile(handle, &session, m_UploadFileName);
		
	  	if (obexDisconnect(handle))
	  	{
			QMessageBox::warning( this, 
							tr("Warning"),
							tr("OBEX disconnect error")
							);
	    	return;
  		}
		
		setLedGreen();					
	}
	else
	{
  		QMessageBox::warning( this, 
						tr("Warning"),
						tr("Phone is not connected")
						);
	}
}


/**
  * Dieser Slot dient zum Auswaehlen der Datei, die an das Telphon geschickt werden soll. 
  */
void BluePhone::fileBrowse()
{
		/* Zeige einen "Open File" Dialog */
		m_UploadFilePath = QFileDialog::getOpenFileName(
							tr("/home"),
							tr("All files (*.*)"),
    	          	     	this,
    	               		tr("Open File"),
							tr("Open File to Upload")
					   		);
		
		lineEditFilePath->setText(m_UploadFilePath);
		
		/* Passe den Namen (Pfad) fuer das Telephon kompatbel (nur Datainame) an */
		if (m_UploadFilePath.contains("/"))
  		{
			m_UploadFileName = m_UploadFilePath.right(m_UploadFilePath.length() - m_UploadFilePath.findRev("/") - 1);
  		}
  		else
  		{
    		m_UploadFileName = m_UploadFilePath;
  		}
		
		lineEditFileNameOnPhone->setText(m_UploadFileName);
		
		/* zeige Informationen zur Datei an */
		showMetaInfo(m_UploadFilePath);
}


/**
  * Dieser Slot liest einige generelle Informationen aus dem Telephon aus.
  * Diese sind: Hersteller, Modell, Revision, Status, Batterieladung
  */
void BluePhone::phoneGetInfo()
{
	if (myLink.getState())
	{					
		/* Hersteller des Telephons abfragen und mit lineEditPhoneManufacturer anzeigen. */
		m_phoneAnswer = myLink.talktoPhone(m_Port, "at+gmi\r");
		m_phoneManufacturer = m_phoneAnswer.join(":");
		m_phoneManufacturer = m_phoneManufacturer.section(':',1,1);
		lineEditPhoneManufacturer->setText(m_phoneManufacturer);
		
		/* Model des Telephons abfragen und mit lineEditPhoneModel anzeigen. */
		m_phoneAnswer = myLink.talktoPhone(m_Port, "at+gmm\r");
		m_phoneModel = m_phoneAnswer.join(":");
		m_phoneModel = m_phoneModel.section(':',1,1);
		lineEditPhoneModel->setText(m_phoneModel);
		
		/* Revision des Telephons abfragen und mit lineEditPhoneRevision anzeigen. */
		m_phoneAnswer = myLink.talktoPhone(m_Port, "at+gmr\r");
		m_phoneRevision = m_phoneAnswer.join(":");
		m_phoneRevision = m_phoneRevision.section(':',1,1);
		m_phoneRevision = m_phoneRevision.left(10);
		lineEditPhoneRevision->setText(m_phoneRevision);
		
		/* Status des Telephons abfragen und mit lineEditPhoneStatus anzeigen. */
		m_phoneAnswer = myLink.talktoPhone(m_Port, "at+cpas\r");
		m_phoneStatus = m_phoneAnswer.join(":");
		m_phoneStatus = m_phoneStatus.section(':',2,2);
		m_phoneStatus = m_phoneStatus.mid(1,1);
		
		if (m_phoneStatus == "0")
		{
			m_phoneStatus = "Ready";
		}
		else if (m_phoneStatus == "1")
		{
			m_phoneStatus = "Unavailable";
		}
		else if (m_phoneStatus == "2")
		{
			m_phoneStatus = "Unknown";
		}
		else if (m_phoneStatus == "3")
		{
			m_phoneStatus = "Ringing";
		}
		else if (m_phoneStatus == "4")
		{
			m_phoneStatus = "Call in progress";
		}
		else if (m_phoneStatus == "5")
		{
			m_phoneStatus = "Asleep";
		}
		else
		{
			m_phoneStatus = "Loathly";
		}
			
		lineEditPhoneStatus->setText(m_phoneStatus);
	
		/* Batterie Ladung des Telephons abfragen und mit progressBarPhoneBattery anzeigen. */
		m_phoneAnswer = myLink.talktoPhone(m_Port, "at+cbc\r");
		m_phoneBattery = m_phoneAnswer.join(":");
		m_phoneBattery = m_phoneBattery.section(':',2,2);
		m_phoneBattery = m_phoneBattery.section(',',1,1);
		progressBarPhoneBattery->setProgress(m_phoneBattery.toInt());
				
		/* Setze den Text des lineEditPhoneConnected auf "Connected" */
		lineEditPhoneConnected->setText("Connected");
		
		/* Die gruene LED soll leuchten */
		setLedGreen();
	}
	else
	{
		/* Setze den Text des lineEditPhoneConnected auf "Disconnected" */
		lineEditPhoneConnected->setText("Disconnected");
		
		/* Loesche den Inhalt aller Felder */
		lineEditPhoneManufacturer->clear();
		lineEditPhoneModel->clear();
		lineEditPhoneRevision->clear();
		lineEditPhoneStatus->clear();
		progressBarPhoneBattery->reset();
		
		/* Die rote LED soll leuchten */
		setLedRed();
					
	}
}


/**
  * Dieser Slot schliesst die Verbindung mit dem Telephon.
  */
void BluePhone::phoneDisconnect()
{
	/* Sind wir verbunden? */
	if (myLink.getState())
	{	
		/* Eventuell mit Daten gefuellte Widgets leeren */
		listViewSms->clear();
		listViewContacts->clear();
				
		/* Serielle Verbindung trennen */
		myLink.disconnectPhone();
		
		/* rfcomm Verbindung trennen */
		myLink.disconnectRfcomm(m_Device);
		
		/* Status Anzeigen leeren */
		phoneGetInfo();
		
	}
	else
	{
		QMessageBox::information( this, 
							tr("Information"),
							tr("Phone is already disconnected")
							);
		setLedRed();
	}
}


/**
  * Dieser Slot verbindet BluePhone mit dem Telephon.
  * Die Logik dieses Slots ist in BlueLink.
  */
void BluePhone::phoneConnect()
{
	/* Sind wir schon schon verbunden? */
	if (myLink.getState())
	{	
		QMessageBox::information( this, 
							tr("Information"),
							tr("Phone is already connected")
							);
		setLedGreen();
	}
	else
	{
		/* Verbinde unser rfcomm Device mit dem entfernten Bluetooth Device (das Telephon). */
		int rfcommState = myLink.connectRfcomm(m_Device, m_BtAddress);
		
		/* rfcomm braucht etwas Zeit. */
		usleep(3000);
		
		if (!rfcommState)
		{
			QMessageBox::critical( this, 
							tr("Error"),
							tr("Unable to connect rfcomm" )
							);
		}
		
	
		/* Baue die serielle (ezV24) Verbindung auf. */
		myLink.connecttoPhone(m_Device);
				
		/* Ermittle den ezv24 Port.*/
		m_Port = myLink.getPort();	
	
		if (m_Port==NULL)
		{
			QMessageBox::critical( this, 
							tr("Error"),
							tr("Could not establish connection to %1\n" 
								"Please try again, sometimes, there are some timing problems" )
                			.arg( m_Device ) 
							);
		}
		else
		{
			phoneGetInfo();
	 	}
	}
}


/**
  * Dieser Slot zeigt die "About" Messagebox an.
  */
void BluePhone::helpAbout()
{
	QMessageBox::about( this, 
							tr("About - BluePhone"),
							tr("BluePhone - Version %1\n\n" 
								"A tool to run a Sony Ericsson mobile phone from a PC\n\n" 
								"(C) 2004 Thomas Gemperli - " 
								"bluephone@gemperli.net\n" )
                			.arg( version ) 
							);
}


/**
  * Diesr Slot beendet das Programm.
  */
void BluePhone::fileExit()
{
	if (myLink.getState())
	{
		phoneDisconnect();
	}
	
	this->close();
}


/** 
  * Dieser Slot setzt die gruene LED rsp. diese "leuchtet", falls die Verbindung zum Telephon steht.
  */
void BluePhone::setLedGreen()
{
	/* Sind wir mit dem Telephon verbunden? */
	if (myLink.getState())
	{
		kLedPhoneStateOrange->off();
		kLedPhoneStateRed->off();
		kLedPhoneStateGreen->on();
	}
	else
	{
		setLedRed();
	}
}

/** 
  * Dieser Slot setzt die orange LED rsp. diese "leuchtet" , falls die Verbindung zum Telephon steht.
  */
void BluePhone::setLedOrange()
{
	/* Sind wir mit dem Telephon verbunden? */
	if (myLink.getState())
	{
		kLedPhoneStateGreen->off();
		kLedPhoneStateRed->off();
		kLedPhoneStateOrange->on();
	}
	else
	{
		setLedRed();
	}
}
	
/** 
  * Dieser Slot setzt die rote LED rsp. diese "leuchtet" 
  */
void BluePhone::setLedRed()
{
	kLedPhoneStateGreen->off();
	kLedPhoneStateOrange->off();
	kLedPhoneStateRed->on();
}


/**
  * Dieser Slot sendet einen neuen oder editierten Kontakt an das Telephon.
  * Parameter: ContactId, ContactType, ContactName, ContactNumber
  */
void BluePhone::sendContactToPhone(const QString& ContactId, const QString& ContactType, const QString& ContactName, const QString& ContactNumber)
{
	
	/* teile dem Telephon mit, dass wir das Telephonbuch des Flash Speichers benutzen wollen. Mehr Infos siehe contactsRead */
	myLink.talktoPhone(m_Port, "at+cpbs=\"ME\"\r");
 
	/* Parameter in die privaten Variablen abspeichern. Wie wir es gelernt haben :) */
	m_phoneContactId = ContactId;
	m_phoneContactType = ContactType;
	m_phoneContactName = ContactName;
	m_phoneContactNumber = ContactNumber;

	/* Der Zeichensatz von m_phoneContactName muss noch fuer das Telephon umgewandelt werden*/
	m_phoneContactName = myConvert.convert_to_gsm(m_phoneContactName);
	
	/* soll ein neuer Kontakt hinzugefuegt oder ein bestehender Kontakt geaendert werden? */
	if (  m_phoneContactId ==  "generate" ) 
	{ 
		m_AtCommand = "at+cpbw=,\"";
	}
  	else
	{
		m_AtCommand = "at+cpbw=";
		m_AtCommand.append(m_phoneContactId);
		m_AtCommand.append(",\"");
	}
	
	/* setze das AT Kommando fuer das Telephon zusammen. Siehe AT Referenz. Zum Glueck gab mir der C++ Gott QString ... */
	m_AtCommand.append(m_phoneContactNumber);
	m_AtCommand.append("\",,\"");
	m_AtCommand.append(m_phoneContactName);
	m_AtCommand.append("/");
	m_AtCommand.append(m_phoneContactType);
	m_AtCommand.append("\"\r");
	
	/* uebermittle dem Telephon den neuen Telephonbuch Eintrag */
	myLink.talktoPhone(m_Port, m_AtCommand);


	/* Lese das Telephonbuch vom Telephon ein. So kann der User ueberpruefen, ob alles geklappt hat */
	contactsRead();
}


/**
  * Dieser Slot sendet ein SMS an das Telephon.
  * Parameter: QString Receiver, QString Message, int index, int id
  */
void BluePhone::sendSmsToPhone(const QString& Receiver, const QString& Message, int index, int id)
{
	
	m_phoneSmsNumber = Receiver;
	m_phoneSmsMessage = Message;
	m_index = index;
	m_id = id;
	
	
	/* Erstelle ein neues BlueConvert Objekt */
	BlueConvert *smsData = new BlueConvert();
	  
	/* Uebersetze die "normalen" uuencode Strings in einen 7Bit Hex PDU Block, den das Telephon benoetigt. */
    smsData->createPduData(m_phoneSmsNumber, m_phoneSmsMessage, m_index, m_id);
	
	/* Hole die erzeugte PDU */  
    m_PDU = smsData->getPdu();
				
	/* Zu new gehoert delete. Ich will ja kein mega Speicher fressendes mini Programm schreiben. */
	delete smsData;
	
	
	/* Ermittle die Laenge der PDU */
	m_PduLength.setNum((m_PDU.length() / 2) - 1); 
	
	/* setze das AT Kommando fuer das Telephon zusammen. Siehe AT Referenz. */
	m_AtCommand = "at+cmgs=";
	m_AtCommand.append(m_PduLength);
	m_AtCommand.append("\r");
	
	/* Nun wird es spassig.
	
	   Eine SMS kann nicht, wie _alle_ anderen Befehle, mit einem einzelnen Kommando verschickt werden. 
	   Nach Abschicken des obrigen Befehls antwortet das Telefon mit folgendem Prompt "> ".
	   In diesem Prompt kann die PDU an das Telefon bergeben werden kann. 
	   Die Sequenz wird mit CTRL-Z (\x1a) abgeschlossen. 
	   Noch was zu CTRL-Z, dies bedeutete unter DOS EOF. Der GSM Standard wurde 1994/95 entwickelt.
	   Der Entwickler scheint DOS benutzt zu haben.
	   
	   An dieser Stelle nochmal einen herzlichen Dank an den Entwickler von ezV24. 
	   Ich wuesste nicht, wie ich ohne seine Lib mit diesem Telephon haette plaudern koennen.
	   Mir kommt z.B. spontan Software Flow Control (XON/XOFF) in den Sinn... */
	
	   
	/* Diese Methode ist im Gesamten etwas speziell, deshalb werden die folgenden drei Variablen, 
	   der Uebersicht halber, gleich hier deklariert. */
	bool prompt = false;
	int stop = 0;
	char buffer[512];

	/* Teile dem Telephon mit, das BluePhone eine PDU codierte SMS senden moechte. */
  	v24Puts(m_Port, m_AtCommand.latin1());
  	while(!stop)
  	{
    	v24Gets(m_Port, buffer, 512);
    	if (strstr(buffer, ">"))
		{
      		stop = 1;
			prompt = true;
		}
    	else if (strstr(buffer, "ERROR"))
    	{
        	stop = 1;
        	QMessageBox::critical( this, 
							tr("Error"),
							tr("Phone is unhappy.\n\nPlease quit BluePhone, reboot your phone and try again.\nThis is not a joke." ) 
							);
    	}
  	}

	/* Falls wir den Prompt haben, uebermittle die PDU */
	if (prompt)
	{
		v24Puts(m_Port, m_PDU.latin1());
  		v24Puts(m_Port, "\x1a");
  		v24Gets(m_Port, buffer, 512);
  		v24Gets(m_Port, buffer, 512);
 
  		if (strstr(buffer, "ERROR"))
		{
        	QMessageBox::critical( this, 
							tr("Error"),
							tr("Phone is unhappy.\n\nPlease quit BluePhone, reboot your phone and try again.\nThis is not a joke." ) 
							);
		}
  		else
		{
         	QMessageBox::information( this, 
							tr("Information"),
							tr("SMS successfully sent." ) 
							);
		}
	}
	else
	{
        QMessageBox::critical( this, 
						tr("Error"),
						tr("Phone does not answer.\n\nPlease quit BluePhone, reboot your phone and try again.\nThis is not a joke." ) 
						);		
	}
}


#include "bluephone.moc"

