/*!
	@class		BlueObex
	@brief		Stellt OpenOBEX zur Verfuegung. Diese "Klasse" ist aus einer OpenOBEX Demo App entstanden.
	@author 	Thomas Gemperli, <bluephone@gemperli.net>
	@version	1.0
	@date		2004-08-08
	@par		This program is free software; you can redistribute it and/or 
				modify it under the terms of the GNU General Public License.
	@file		blueobex.cpp
*/


#include <bluetooth/bluetooth.h>

#include "blueobex.h"

extern "C"
{
  #include <openobex/obex.h>
}

#include <fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>



/**
  * Initialisiert Obex mit allen benoetigen Parametern.
  */
obex_t* obexInit(struct BlueObex *myObex, int method, const char *address, int channel)
{
	int transport_m;

	
	if (method == 1)
	{
    	transport_m = OBEX_TRANS_IRDA;
	}
  	else if (method == 2)
    {
		transport_m = OBEX_TRANS_BLUETOOTH;
	}
  	else
  	{
    	printf("Invalid transport method\n");
    	return NULL;
  	}
    
	obex_t *handle;
	if (! (handle = OBEX_Init(transport_m, obexEvent, 0)))
  	{
  	  printf("Could not init OBEX\n");
 	   return NULL;
	}

	OBEX_SetUserData(handle, myObex);

	if (method == 1)
	{
    	if((IrOBEX_TransportConnect(handle, "OBEX")) < 0 )
    	{
      		printf("OBEX transport connect error\n");
      		return NULL;  
    	}
  	}


	if  (method == 2)
	{
    	bdaddr_t bda;

		str2ba(address, &bda);

    	if(BtOBEX_TransportConnect(handle, BDADDR_ANY, &bda, channel) < 0)
    	{
			printf("OBEX transport connect error\n");
			return NULL;
		}
	}

    return handle;
}


/**
  * Stellt eine Obex Verbindung zur Gegenstelle her.
  */
int obexConnect(obex_t *handle)
{
	obex_object_t *object;
	obex_headerdata_t header;

	if (! (object = OBEX_ObjectNew(handle, OBEX_CMD_CONNECT)))
	{
		printf("Could not create OBEX object\n");
		return -1;
	}

	header.bs = (uint8_t*)"Linux";

	if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_WHO, header, 6, OBEX_FL_FIT_ONE_PACKET) < 0)
	{
		printf("Could not add OBEX header\n");
		OBEX_ObjectDelete(handle, object);
		return -1;
	}
  
	OBEX_Request(handle, object);
	syncwait(handle);
	
	return 0;
}


/**
  * Behandelt ein Obex Ereigniss.
  */
void obexEvent(obex_t *handle, obex_object_t *object, int mode, int event, int obex_cmd, int obex_rsp)
{
	switch (event)
	{
		case OBEX_EV_PROGRESS:
			printf("OBEX made some progress..\n");
			break;

		case OBEX_EV_ABORT:
			printf("OBEX request aborted\n");
			break;

		case OBEX_EV_REQDONE:
			obexEventDone(handle, object, obex_cmd, obex_rsp);
			break;

		case OBEX_EV_REQHINT:
			OBEX_ObjectSetRsp(object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
			break;

		case OBEX_EV_LINKERR:
			OBEX_TransportDisconnect(handle);
			printf("OBEX link broken\n");
			break;

		default:
			printf("Unknown OBEX event\n");
			break;
	}
}


/**
  * Liest eine Datei in einen Buffer ein und stellt sie somit Obex zur Verfuegung, dass sie gesendet werden kann.
  */
char* obexPutReadFile(const char *filename, int *length)
{
	int fd;
	char *file_buf = NULL;
	struct stat stat_buf;

	fd = open(filename, O_RDONLY);
	if (fd == -1)
	{
    	printf("Error opening file: %s\n", filename);
	}
	else
	{
		if (fstat(fd, &stat_buf) == -1)
		{
			printf("Could not stat file: %s\n", filename);
		}
		else
		{
			*length = stat_buf.st_size;

			file_buf = (char*)malloc(*length);

			if (file_buf == NULL)
			{
				printf("Malloc failed\n");
				file_buf = NULL;
				*length = 0;
			}
			else
			{
				int nb = read(fd, file_buf, *length);
				
				if (nb != *length)
				{
					printf("Error: short read\n");
					exit(1);
				}
			}
		}
		
		close(fd);
	}
	
	return file_buf;
}


/**
  * Sendet der Gegenstelle eine Datei.
  */
int obexPutfile(obex_t *handle, struct BlueObex *myObex, const char *name)
{
	obex_object_t *object;
	obex_headerdata_t header;
	int bodysize;
	char *body;
	uint8_t *tmp;
	int namelen;

	if (!(object = OBEX_ObjectNew(handle, OBEX_CMD_PUT)))
	{
		printf("Error creating OBEX object\n");
		return -1;  
	}

	body = obexPutReadFile(myObex->filename, &bodysize);

	if (body == NULL)
	{
		OBEX_ObjectDelete(handle, object);
		return -1;
	}

	namelen = (strlen(name)*2)+2;
	tmp = (uint8_t*)malloc(namelen);

	if(header.bs)
	{
		namelen = OBEX_CharToUnicode(tmp, (uint8_t*)name, namelen);
		header.bs = tmp;

		if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_NAME, header, namelen, 0) < 0)
		{
			printf("Error add OBEX name header\n");
			OBEX_ObjectDelete(handle, object);
			free(body);
			return -1;
		}
		
		free ((void*)header.bs);
	}
	else
	{
		printf("Malloc error\n");
		return -1;
	}

	header.bq4 = bodysize;
	
	if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_LENGTH, header, 4, 0) <0)
	{
		printf("Error adding OBEX length header\n");
		OBEX_ObjectDelete(handle, object);
		free(body);
		return -1; 
	}

	header.bs = (uint8_t*)body;
	
	if(OBEX_ObjectAddHeader(handle, object, OBEX_HDR_BODY, header, bodysize, 0) <0 )
	{
		printf("Error adding OBEX body header\n");
		OBEX_ObjectDelete(handle, object);
		free(body);
		return -1;
	}

	OBEX_Request(handle, object);
	syncwait(handle);

	free(body);
	return 0;
}


/**
  * Behandelt das Warten auf die Bestaetigung der Gegenstelle.
  */
void syncwait(obex_t *handle)
{
	struct BlueObex *session;
	int ret;

	session = (BlueObex*)OBEX_GetUserData(handle);

	while (!session->client_done)
	{
		ret = OBEX_HandleInput(handle, 10);
		
		if (ret < 0)
		{
			printf("Error on OBEX HandleInput\n");
			break;
		}
		else if (ret == 0)
		{
			printf("OBEX timeout\n");
			OBEX_CancelRequest(handle, false);
			break;
		}
	}
	
	session->client_done = false;
}


/**
  * Beendet ein Obex Ereigniss.
  */
void obexEventDone(obex_t *handle, obex_object_t *object, int obex_cmd, int obex_rsp)
{
	struct BlueObex *session;

	session = (BlueObex*)OBEX_GetUserData(handle);

	switch (obex_cmd)
	{
    
		case OBEX_CMD_CONNECT:
		{
			if (obex_rsp == OBEX_RSP_SUCCESS)
			{
				printf("OBEX connected\n");
			}
			else
			{
				printf("OBEX connect failed\n");
			}
			
			break;
		}

		case OBEX_CMD_DISCONNECT:
		{
			if (! (object = OBEX_ObjectNew(handle, OBEX_CMD_DISCONNECT)))
			{
				printf("Error on OBEX disconnect\n");
				return;
			}
			
			OBEX_Request(handle, object);
			syncwait(handle);
			
			break;
		}

		case OBEX_CMD_PUT:
		{
			if (obex_rsp == OBEX_RSP_SUCCESS)
			{	
				printf("OBEX PUT succeeded\n");
			}
      		else
			{	
				printf("OBEX PUT failed\n");
			}
			
			break;
		}

		default:
			break;
	}
	
	session->client_done = true;
}


/**
  * Trennt eine Obex Verbindung.
  */
int obexDisconnect(obex_t *handle)
{
	OBEX_TransportDisconnect(handle);
	
	printf("OBEX disconnected\n");
	
	return 0;
}


