/*!
	@class		BlueLink
	@brief		Stellt die Verbindung zum Telephon bereit.
	@author 	Thomas Gemperli, <bluephone@gemperli.net>
	@version	1.0
	@date		2004-08-03
	@par		This program is free software; you can redistribute it and/or 
				modify it under the terms of the GNU General Public License.
	@file		bluelink.cpp
*/


#include "bluelink.h"

//FORTESTNG
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>


/**
  * BlueLink Konstruktor
  * Erstellt ein BlueLink Objekt.
  */
BlueLink::BlueLink()
{
	m_port = NULL; 
	m_communicate = false;
	m_connected = false;

}


/**
  * BlueLink Destruktor
  */
BlueLink::~BlueLink()
{
}


/** 
  * Diese Methode liefet einen Pointer auf einen v24_port_t zurueck.
  * Siehe http://ezv24.sourceforge.net/api-html fuer weitere Informationen.
  */
v24_port_t* BlueLink::getPort()
{
	return m_port;
}


/** 
  * Diese Methode liefert den Status des Links zum Telephon zurueck.  
  * Rueckgabewert: bool Status
  */
bool BlueLink::getState()
{
	return m_connected;
}


/** 
  * Diese Methode verbindet das rfcomm Device mit dem entfernten Bluetooth Device (dem Telephon). 
  * Parameter: const QString& comDevice, btAddr
  */
int BlueLink::connectRfcomm(const QString& comDevice, const QString& btAddr)
{
	/* setze das rfcomm Kommando zusammen. */
	m_rfcommCommand = "/usr/bin/rfcomm connect ";
	m_rfcommCommand.append(comDevice);
	m_rfcommCommand.append(" ");
	m_rfcommCommand.append(btAddr);
	m_rfcommCommand.append(" 1");
	
	/* execve will Char Arrays sehen, keine QStrings. Deshalb die Umwandlung hier. */
	const char *commandChar = m_rfcommCommand.latin1();
	
	pid_t child_pid;
	int child_status;

	child_pid = fork();
	
	if(child_pid == 0)
	{
		/* Dieser Teil wird vom Kind Prozess ausgefuehrt. */
    
		execl("/bin/sh", "sh", "-c", commandChar, 0);
		
		/* Falls execve zurueckkommt, ist etwas schiefgegangen. */
		printf("Unknown command\n");
		exit(0);
	}
	else 
	{
    	/* Dieser Teil wird vom Vater Prozess (BluePhone) ausgefuehrt. */
		 
		return child_status;
	}
}


/** 
  * Diese Methode trennt die rfcomm Verbindung mit dem Telephon. 
  * Parameter: const QString& comDevice
  */
int BlueLink::disconnectRfcomm(const QString &comDevice)
{
	/* setze das rfcomm Kommando zusammen. */
	m_rfcommCommand = "/usr/bin/rfcomm release ";
	m_rfcommCommand.append(comDevice);

	/* execve will Char Arrays sehen, keine QStrings. Deshalb die Umwandlung hier. */
	const char *commandChar = m_rfcommCommand.latin1();
	
	pid_t child_pid;
	int child_status;

	child_pid = fork();
	
	if(child_pid == 0)
	{
		/* Dieser Teil wird vom Kind Prozess ausgefuehrt. */
    
		execl("/bin/sh", "sh", "-c", commandChar, 0);
		
		/* Falls execve zurueckkommt, ist etwas schiefgegangen. */
		printf("Unknown command\n");
		exit(0);
	}
	else 
	{
    	/* Dieser Teil wird vom Vater Prozess (BluePhone) ausgefuehrt. */
		 
		return child_status;
	}
}


/**
  * Diese Methode erstellt die Verbindung zum Telephon.
  * Parameter: QString mit dem zu verwendenden Device, diese sind unter Linux normalerweise:
  * /dev/rfcomm0 fuer bluetooth, /dev/ircomm0 fuer IrDA oder /dev/ttyS0 fuer ein serielles Kabel
  */
void BlueLink::connecttoPhone(const QString &comDevice)
{
	m_communicate = false;
	m_connected = false;
		
	
	m_communicate = true;
	/* ezv24 Port oeffnen. Siehe http://ezv24.sourceforge.net/api-html */
	m_port = v24OpenPort(comDevice, V24_RTS_CTS);
	m_communicate = false;
	
	/* Konnte der Port geoeffnet werden?  */
	if (m_port==NULL)
	{
		m_connected = false;
	}
	else
	{
    	m_connected = true;
				
		/* "New SMS Benachrichtigungs Modus" des Telephons abfragen. 
		 * Siehe AT Kommando Referenz Seite 93ff. 
		 */
		m_phoneAnswer = talktoPhone(m_port, "at+cnmi=?\r");
	    m_phoneMode = m_phoneAnswer.join(":");
		m_phoneMode = m_phoneMode.section(':',2,2);
		m_phoneMode = m_phoneMode.section('(',1,1);
		m_phoneMode = m_phoneMode.section(')',0,0);

		/* Setze einen passenden Modus. T68/T610 sind gleich (2) */
	    if (m_phoneMode == "2")
		{
    		m_phoneAnswer = talktoPhone(m_port, "at+cnmi=2,1,0,0\r");
		}
    	else if (m_phoneMode == "3")
		{
	    	m_phoneAnswer = talktoPhone(m_port, "at+cnmi=3,1,0,0\r");
		}
  	}
}


/**
  * Diese Methode trennt die Verbindung zum Telephon.
  */
void BlueLink::disconnectPhone()
{
	
	v24ClosePort(m_port);
	m_port = NULL; 
	m_communicate = false;
	m_connected = false;

}


/**
  * Diese Methode tauscht Informationen mit dem Telephon aus.
  * Parameter: v24_port_t* v24_port, QString Befehl .
  * Rueckgabewert: QStringList mit allen Antworten des Telephons.
  */	
QStringList BlueLink::talktoPhone(v24_port_t *m_port, const QString &m_command)
{
	/* 
	  * Beinhaltet die Antwort des Telephons. 
	  * Kann nicht im Header stehen, da wir den ersten Wert nicht zuweisen, sondern "hineinstreamen" (<<)
	  * Wo genau der Fehler ist, weiss ich nicht (QT, Compiler?), auf alle Faelle bestaetigt der Debugger obriges.
	  */
	QStringList m_phoneTalk;
	/* Dasselbe gilt fuer die hier. */
	QString m_phoneTmp;
	
	/* Um die Ubersicht zu behalten, ist auch diese Variable nicht im Header definiert.
	 * -> Alle Variablen dieser Methode sind hier im .cpp deklariert.
	 */
	int m_v24Value = 0;
	
	/* Ein 80 Byte grosser Char Array, der den Rueckgabestring von v24Gets enthalten wird. */
	char ansChar[80];
	
	/* ezV24 will Char Arrays sehen, keine QStrings. Deshalb die Umwandlung hier. */
	const char *cmdChar = m_command.latin1();
	
	
	/* Wir beginnen zu kommunizieren */
	m_communicate = true;
	
	/* Sende den Command String */
	m_v24Value = v24Puts(m_port, cmdChar);
	
	/* Wenn der Rueckgabewert von v24Puts kleiner ist, als die Laenge des uebergebenen Kommandos, ist etwas schiefgegangen. */
	if (m_v24Value < strlen(cmdChar))
	{
		m_communicate = false;
		m_phoneTalk << "ezv24Error";     
		return m_phoneTalk;
	}
	else
	{
		while((strncmp(ansChar, "OK", 2) != 0) && (strncmp(ansChar, "ERROR", 5) != 0))
		{
			/* Empfange einen String. */
			m_v24Value = v24Gets(m_port, ansChar, sizeof(ansChar)-1);
			
			/* v24Gets liefert -1 zurueck, wenn etwas schiefgegangen ist. */
			if (m_v24Value == -1)
			{
				m_communicate = false;
				m_phoneTalk << "ezv24Error";
				return m_phoneTalk;
			}
			else
			{
				/* Durchsuche die Antwort nach 0en. Diese werden von Telephonen (GSM) als @ verwendet. Siehe AT Kommando Referenz. */
				for (int i = 0; i < m_v24Value; i++)
        			if (!ansChar[i])
					{
         				ansChar[i] = 0x80;
					}
      				
					m_phoneTmp.append(ansChar);
					
       				if (m_phoneTmp.right(1) == "\n")
        			{
        				m_phoneTalk << m_phoneTmp;
         				m_phoneTmp = "";
        			}
      		}
    	}
    
		m_communicate = false;
    	return m_phoneTalk;
  	} 

}


