/*!
	@class		BlueConvert
	@brief		Konvertiert 7bit GSM Zeichensaetze vom Telephon und zurueck.
	@author 	Thomas Gemperli, <bluephone@gemperli.net>
	@version	1.0
	@date		2004-08-03
	@par		This program is free software; you can redistribute it and/or 
				modify it under the terms of the GNU General Public License.
	@file		blueconvert.h
*/


#ifndef BLUECONVERT_H
#define BLUECONVERT_H


#include <qstring.h>



/**
  * Diese Klasse dient der Umwandlung der 7Bit GSM Zeichensaetze vom Telephon zu PC Unicode und umgekehrt, 
  * weiter der Umwandlung des "ERICSSON" Zeichensatzes zu PC Unicode und ebenfalls auch umgekehrt.
  * Sie stellt ebenfalls Methoden zur codierung oder decodierung eines SMS PUD zur Verfuegung.
  * Einige Methoden (v.a. die riesen Cases) dieser Klasse sind uebernommen von scmxx, 
  * einer aelteren Suite von Terminalprogrammen fuer aeltere Simens Mobil Telephone.
  * scmxx wurde von Hendrik Sattler geschrieben und unter GPL veroeffentlicht.
  * Die Programme sowie deren Source sind unter http://www.hendrik-sattler.de/scmxx/ verfuegbar.
  * Die uebernommenen Methoden habe ich an meine Beduerfnisse angepasst.
  */
class BlueConvert{

public:

	/**
   	  * Erstellt ein BlueConvert Objekt.
  	  */
    BlueConvert();
	
	
	/**
  	  * Destruktor
   	  */
    ~BlueConvert();
	
	
	/** 
	 * Diese Methode konvertiert den GSM Zeichensatz in Unicode.
	 * Wird benoetigt, um das Telephonbuch vom Telephon zu lesen.
	 * Parameter: QString input
	 * Rueckgabewert: QString UnicodeString
	 */
	QString convert_from_gsm (const QString& input);
	
	
	/** 
	 * Diese Methode konvertiert Unicode in den GSM Zeichensatz.
	 * Wird benoetigt, um das Telephonbuch auf das Telephon zu schreiben.
	 * Parameter: QString input
	 * Rueckgabewert: QString GSMString
	 */
	QString convert_to_gsm (const QString& input);
	
	
	/** 
	 * Diese Methode konvertiert das GSM 7-Bit Hex PDU Format in Unicode.
	 * Wird benoetigt, um SMS vom Telephon zu lesen.
 	 * Parameter: QString input
	 * Rueckgabewert: QString decodedMessage
 	 */
	QString decodeMessage(QString encodedMessage);
	
	
	/** 
 	 * Diese Methode konvertiert Unicode in das GSM 7-Bit Hex PDU Format.
	 * Wird benoetigt, um SMS auf das Telephon zu schreiben rsp. zu versenden.
 	 * Parameter: QString input, int Length
	 * Rueckgabewert: QString encodedMessage
	 */
	QString encodeMessage(QString decodedMessage, int longMessage);
	
	
	/** 
 	 * Diese Methode extrahiert die einzelnen Daten aus dem PDU Format.
	 * Ist v.a. String Akrobatik. Eine gute Erklaerung zum Thema befindet sich da:
	 * http://www.nobbi.com/sms_pdu.htm
 	 * Parameter: QString input, int Length
	 */
	void extractPduData(QString PDU);
	
	
	/** 
 	 * Diese Methode erstellt einen PDU Datenblock.
	 * Ist v.a. String Akrobatik. Eine gute Erklaerung zum Thema befindet sich da:
	 * http://www.nobbi.com/sms_pdu.htm
 	 * Parameter: QString recipient, QString message, int index, int id
	 */
 	void createPduData(QString recipient, QString message, int index, int id);
  
	
	/** 
 	 * Diese Methode swappt Bits.
	 * Einige Daten (z.B. smsc Nummer) in einer PDU sind BCD codiert.
 	 * Parameter: QString toSwap
	 * Rueckgabewert: QString swapped
	 */
  	QString decodeBCD(QString toSwap);
	
	
	/** 
 	 * Diese Methode liefert den decodierten Sender einer SMS zurueck.
	 * Rueckgabewert: QString m_sender
	 */	
	QString getSender();
	
	
	/** 
 	 * Diese Methode liefert das decodierte Datum einer SMS zurueck.
	 * Rueckgabewert: QString m_date
	 */	
	QString getDate();
	
	
	/** 
 	 * Diese Methode liefert die decodierte Nachricht einer SMS zurueck.
	 * Rueckgabewert: QString m_message
	 */	
	QString getMessage();
	
	
	/** 
 	 * Diese Methode liefert eine encodierte PDU zurueck.
	 * Rueckgabewert: QString m_PDU
	 */	
	QString getPdu();
		

	
private:
 	
  	/** Enthaelt die PDU Daten (GSM 7-Bit Hex) als QString.
	 * Ein PDU Block sieht z.B. folgendermassen aus: 
	 * 079194712272303325000C9194711232547600000BD4F29C4E2FE3E9BA4D19
	 * Siehe auch den Kommentar zu extractPduData.
	 */
	QString m_PDU;
	
	/** Enthaelt die Absendernummer der Nachricht. */
	QString m_sender;
	
	/** Enthaelt die eigentliche Nachricht. */
	QString m_message;
	
	/** Enthaelt das Empfangsdatum der Nachricht. */
	QString m_date;

  	/** Enthaelt die aktuelle Position der Verarbeitung im PDU. */
	int m_posInPdu;
 
	/** Enthaelt den Typ der SMSC-Adresse. */
	int m_smscType;
	
	/** Enthaelt die Lnge der SMSC-Adresse. */
	int m_smscLen;
  
  	/** Enthaelt die Lnge der Absendernummer. */
	int m_senderLength;
	
	/** Enthaelt den Typ der Absendernummer. */
	QString m_senderType;
	
	/** Enthaelt die Laenge des User Data Headers. */
	int m_udhLength;
  
	/** Enthaelt Informationen zum Reply Header. */
	int m_rp;
	
	/** Enthaelt die Laenge des Reply Headers. */
	int m_rplength;
	
	/** Hilfsvariable. */
	int m_ie[8];
	
	/** Enthaelt die Lnge der eigentlichen Nachricht. */
	int m_messageLength;

	
	/** Enthaelt den BCD swap String. */
	QString m_swapped;
	
	
	/** Enthaelt die Empfaengernummer einer Nachricht. */
	QString m_recipient;
	
	/** Der Index auf dem Telephon einer Nachricht. */
	int m_index;
	
	/** Die ID einer Nachricht. */
	int m_id;
	
	/** Falls eine ueberlange Nachricht gesendet wird, enthaelt diese Variable die Ueberlaenge. */
	int m_longMessage;
	
	/** Enthaelt den Typ der Empfaengernummer (National/International). Default ist 85 (National). */
	QString m_recipientType;

	/** Enthaelt die Laenge der Empfaengernummer. */
	int m_recipientLength;
	
	/** Enthaelt die Laenge der Empfaengernummer in Hex. */
	QString m_recipientLengthHex;
	
	/** Enthaelt die Laenge des Restes (>160) einer Nachricht bei einer ueberlangen Nachricht. */
	QString m_carryoverLength;
	
	/** Enthaelt die eigentliche Nachricht m PDU Format. */
	QString m_encodedMessage;
	

};


#endif


