/*!
	@class		BlueConvert
	@brief		Konvertiert 7bit GSM Zeichensaetze vom Telephon und zurueck.
	@author 	Thomas Gemperli, <bluephone@gemperli.net>
	@version	1.0
	@date		2004-08-03
	@par		This program is free software; you can redistribute it and/or 
				modify it under the terms of the GNU General Public License.
	@file		blueconvert.cpp
*/


#include "blueconvert.h"


/**
  * BlueConvert Konstruktor
  * Erstellt ein BlueConvert Objekt.
  */
BlueConvert::BlueConvert()
{
}


/**
  * BlueConvert Destruktor
  */
BlueConvert::~BlueConvert()
{
}


/** 
 * Diese Methode liefert den decodierten Sender einer SMS zurueck.
 * Rueckgabewert: QString m_sender
 */	
QString BlueConvert::getSender()
{
	return m_sender;
}
	
	
/** 
 * Diese Methode liefert das decodierte Datum einer SMS zurueck.
 * Rueckgabewert: QString m_date
 */	
QString BlueConvert::getDate()
{
	return m_date;
}
	
	
/** 
 * Diese Methode liefert die decodierte Nachricht einer SMS zurueck.
 * Rueckgabewert: QString m_message
 */	
QString BlueConvert::getMessage()
{
	return m_message;
}
	
	
/** 
 * Diese Methode liefert die encodierte PDU zurueck.
 * Rueckgabewert: QString m_PDU
 */	
QString BlueConvert::getPdu()
{
	return m_PDU;
}


/** 
  * Diese Methode extrahiert die einzelnen Daten aus dem PDU Format.
  * Ist v.a. String Akrobatik. Eine gute Erklaerung zum Thema befindet sich da:
  * http://www.nobbi.com/sms_pdu.htm
  * Parameter: QString input, int Length
  */
void BlueConvert::extractPduData(QString PDU)
{
	m_posInPdu = 0;
  	
	m_PDU = PDU;
  	
	/* Extrahiere allgemeine Daten des PDU Blockes wie 
	   die Lnge der SMSC-Adresse (Service Center und Typ dieser (National/International) */
	m_smscLen = m_PDU.left(2).toInt(0, 16);
  	m_smscType = m_PDU.mid((m_smscLen*2)+2,2).toInt(0,16);
  	
	/* Weiter im PDU Block */
	m_posInPdu = (m_smscLen*2)+4;
  	if (m_smscType&1) 
	{	
  		m_posInPdu += 2; 
	}
	
	/* Extrahiere die Laenge der Sendernummer */
 	m_senderLength = (m_PDU.mid(m_posInPdu,2)).toInt(0,16);
	
	/* Extrahiere den Typ der Telephonnummer (National/International) */
  	m_senderType = (m_PDU.mid(m_posInPdu+2,2));
  	if (m_senderLength % 2)
	{
		m_senderLength++;
	}
  	
	/* Extrahiere die Nummer des Senders. Die erste wirklich interessante Information */
  	m_sender = m_PDU.mid(m_posInPdu+4, m_senderLength);
  	m_sender = decodeBCD(m_sender);
  	if (m_sender.contains("F"))
    {
		m_sender.truncate(m_sender.length()-1);
	}
	
	if (m_senderType=="91")
	{
		m_sender.prepend("+");
	}
	
  	m_posInPdu +=4+m_senderLength;
	

  	/* Die naechsten 4 Bits interssieren uns nicht. (Protocol Identifier & Data Coding Scheme) */
  	m_posInPdu +=4;
	
	/* Extrahiere das Sendedatum des SMS */
  	if (!(m_smscType&1))
  	{
	  	m_date = m_PDU.mid(m_posInPdu, 14);  
	  	m_date = decodeBCD(m_date);
	  	 
	  	m_posInPdu +=14;
  	}
 	else if (!(m_smscType&64))
  	{
		m_posInPdu += 2;
	}
   	
	/* Stelle fest, ob es sich um eine ueberlange SMS handelt (mehr als 160 Zeichen)
	   Und setze m_posInPdu entsprechend */
	m_messageLength = (m_PDU.mid(m_posInPdu, 2)).toInt(0,16);
	m_posInPdu +=2;
	
	if (m_smscType&64) 
	{
		m_udhLength=m_PDU.mid(m_posInPdu, 2).toInt(0,16);
		m_posInPdu +=2;
		m_rp=m_PDU.mid(m_posInPdu, 2).toInt(0,16);
		m_posInPdu +=2;
		m_rplength=m_PDU.mid(m_posInPdu, 2).toInt(0,16);
		m_posInPdu +=2;
		
		for (int i=0;i<m_rplength;i++)
		{
				m_ie[i]=m_PDU.mid(m_posInPdu, 2).toInt(0,16);
				m_posInPdu +=2;
		}
		m_posInPdu-=2*(m_udhLength+1);
	}
	
	/* Extrahiere die eigentliche Nachricht des SMS */
	m_message = m_PDU.mid(m_posInPdu);
 	m_message = decodeMessage(m_message);
	if (m_smscType&64)
	{
		int toskip=(m_udhLength+1)*8/7;
		
		if ((m_udhLength+1)*8%7)
		{
			toskip++;
		}
		
		m_message=m_message.mid(toskip);
	}
}


/** 
  * Diese Methode swappt Bits.
  * Einige Daten (z.B. smsc Nummer) in einer PDU sind BCD codiert.
  * Parameter: QString toSwap
  * Rueckgabewert: QString swapped
  */
QString BlueConvert::decodeBCD(QString toSwap)
{
	m_swapped = "";

  	for (unsigned int i=0; i<=toSwap.length(); i+=2)
  	{
	    m_swapped.append(toSwap.mid(i+1,1));
	    m_swapped.append(toSwap.mid(i,1));
  	}
  
  	return m_swapped;   
}


/** 
  * Diese Methode erstellt einen PDU Datenblock.
  * Ist v.a. String Akrobatik. Eine gute Erklaerung zum Thema befindet sich da:
  * http://www.nobbi.com/sms_pdu.htm
  * Parameter: QString recipient, QString message, int index, int id
  */
void BlueConvert::createPduData(QString recipient, QString message, int index, int id)
{
	m_recipient = recipient;
	m_message = message;
	m_index = index;
	m_id = id;
	
	/* Haben wir eine ueberlange Message? 
	   Wenn ja, wird die Groesse dieser Ueberlaenge spaeter gebraucht */
	m_longMessage=m_message.length()>160;
	
	/* Eine ueberlange Message hat einen anderen Header als eine "nornale" */
	m_PDU = m_longMessage?"004100":"000100";
	
	/* Setze den Typ der Empfaengernummer. (National) */
	m_recipientType = "85";
	
	/* Ermittle den Typ der Empfaengernummer. Falls International, setze dies (91). */
  	if (m_recipient.left(1)=="+")
  	{
	    m_recipient=m_recipient.mid(1);
	    m_recipientType = "91";
  	}
	
	/* Ermittle die Laenge der Empfaengernummer. */
	m_recipientLength = m_recipient.length();
 	m_recipientLengthHex.setNum(m_recipientLength,16);
  	if (m_recipientLength < 16)
    {
		m_PDU.append("0");
  	}
    m_PDU.append(m_recipientLengthHex);
	
  	/* Setze den Typ der Empfaengernummer. (National/International) */
  	m_PDU.append(m_recipientType);
	
	/* Fuege die BCD encodierte Empfaengernummer der PDU hinzu */
  	if ((m_recipientLength%2) != 0)
	{
    	m_recipient.append("F");
	}
  	m_PDU.append(decodeBCD(m_recipient));
	
	/* Fuege Protokoll Id and Data Coding Scheme der PDU hinzu. Die sind immer gleich (beide 00). */
  	m_PDU.append("0000");

  	/* Konvertiere die Nachricht in das PDU Format. */
  	if (m_longMessage)
	{
  		m_encodedMessage = encodeMessage(m_message.mid(153*(m_index-1),153*m_index),m_longMessage);
	}
  	else
	{
  		m_encodedMessage = encodeMessage(m_message,m_longMessage);
	}

	/* Falls es sich um eine ueberlange Nachricht handelt, ermittle die Ueberlaenge
	   und setze die Gesamtlaenge entsprechend. */
	if (m_longMessage)
  	{
  		m_carryoverLength.setNum(m_messageLength+7, 16);
  		
		if (m_messageLength < 16)
		{
    		m_carryoverLength.prepend("0");
		}
		
  		m_PDU.append(m_carryoverLength);
    	m_PDU.append("050003");
		
    	int chars=m_message.length();
    	int numSMS=chars/153;
    	
		if (chars%153)
		{
    		numSMS++;
		}
    	
		m_carryoverLength.setNum(m_id, 16);
		
    	if (m_id<16)
		{
    		m_carryoverLength.prepend("0");
		}
    	m_PDU.append(m_carryoverLength);
    	
		m_carryoverLength.setNum(numSMS, 16);
    	
		if (numSMS<16)
		{
    		m_carryoverLength.prepend("0");
		}
		
   		m_PDU.append(m_carryoverLength);
    	
		m_carryoverLength.setNum(m_index, 16);
    	
		if (m_index<16)
		{
    		m_carryoverLength.prepend("0");
		}
		
    	m_PDU.append(m_carryoverLength);
	}
  	else
  	{
  		m_carryoverLength.setNum(m_messageLength, 16);
  		
		if (m_messageLength < 16)
		{
    		m_carryoverLength.prepend("0");
		}
		
		m_PDU.append(m_carryoverLength);
	}

  /* Fuege die Nachricht der PDU hinzu. Fertig ist das SMS (in m_PDU). */
  m_PDU.append(m_encodedMessage);

}


/** 
  * convert_from_gsm
  * Diese Methode konvertiert den GSM Zeichensatz in Unicode.
  * Parameter: QString input
  * Rueckgabewert: QString UnicodeString
  */
QString BlueConvert::convert_from_gsm (const QString& input)  
{
    QString retval;
    for (uint i = 0;i < input.length(); i++) {
  	const ushort& a = input[i].unicode();
	switch (a) {
	    default:
		retval += a;
		break;
	    case 0x80: // commercial at
		/*
		 * we expect this as 0x80 instead of 0x00
		 * to avoid problem with C's char*
		 */
		retval +=  0x0040;
		break;
	    case 0x01: // pound sign
		retval +=  0x00a3;
		break;
	    case 0x02: // dollar sign
		retval +=  0x0024;
		break;
	    case 0x03: // yen sign
		retval +=  0x00a5;
		break;
	    case 0x04: // small 'e' with grave
		retval +=  0x00e8;
		break;
	    case 0x05: // small 'e' with acute
		retval +=  0x00e9;
		break;
	    case 0x06: // small 'u' with grave
		retval +=  0x00f9;
		break;
	    case 0x07: // small 'i' with grave
		retval +=  0x00ec;
		break;
	    case 0x08: // small 'o' with grave
		retval +=  0x00f2;
		break;
	    case 0x09: // small/capital 'c' with cedilla (we map to small)
		retval +=  0x00e7; // or 0x00c7 ?
		break;
	    case 0x0b: // capital 'O' with stroke
		retval +=  0x00d8;
		break;
	    case 0x0c: // small 'o' with stroke
		retval +=  0x00f8;
		break;
	    case 0x0e: // capital 'A' with ring above
		retval +=  0x00c5;
		break;
	    case 0x0f: // small 'a' with ring above
		retval +=  0x00e5;
		break;
	    case 0x10: // capital delta
		retval +=  0x0394;
		break;
	    case 0x11: // underscore / low line
		retval +=  0x005f;
		break;
	    case 0x12: // capital phi
		retval +=  0x03a6;
		break;
	    case 0x13: // capital gamma
		retval +=  0x0393;
		break;
	    case 0x14: // capital lambda
		retval +=  0x039b;
		break;
	    case 0x15: // capital omega
		retval +=  0x03a9;
		break;
	    case 0x16: // capital pi
		retval +=  0x03a0;
		break;
	    case 0x17: // capital psi
		retval +=  0x03a8;
		break;
	    case 0x18: // capital sigma
		retval +=  0x03a3;
		break;
	    case 0x19: // capital theta
		retval +=  0x0398;
		break;
	    case 0x1a: // capital xi
		retval +=  0x039e;
		break;
	    case 0x1b:
		++i;
		switch (a) {
		    default: // non-breaking space
			--i;
			retval +=  0x00a0;
			break;
		    case 0x1a: // form feed
			retval +=  0x000c;
			break;
		    case 0x14: // circumflex accent
			retval +=  0x005e;
			break;
		    case 0x28: // left curly bracket
			retval +=  0x007b;
			break;
		    case 0x29: // right curly bracket
			retval += 0x007d;
			break;
		    case 0x2f: // reverse solidus
			retval += 0x005c;
			break;
		    case 0x3c: // left square bracket
			retval += 0x005b;
			break;
		    case 0x3d: // tilde
			retval += 0x007e;
			break;
		    case 0x3e: // right square bracket
			retval += 0x005d;
			break;
		    case 0x40: // vertical line
			retval += 0x007c;
			break;
		    case 0x65: // Euro sign
			retval += 0x20ac;
			break;
		}
		break;
	    case 0x1c: // capital 'AE'
		retval +=  0x00c6;
		break;
	    case 0x1d: // small 'ae'
		retval +=  0x00e6;
		break;
	    case 0x1e: // small sharp s (german)
		retval +=  0x00df;
		break;
	    case 0x1f: // capital 'E' with acute
		retval +=  0x00c9;
		break;
	    case 0x24: // currency sign
		//retval +=  0x20ac;
		retval +=  0x00a4;
		break;
	    case 0x40: // inverted exclamation mark
		retval +=  0x00a1;
		break;
	    case 0x5b: // capital 'A' with diaeresis
		retval +=  0x00c4;
		break;
	    case 0x5c: // capital 'O' with diaeresis
		retval +=  0x00d6;
		break;
	    case 0x5d: // capital 'N' with tilde
		retval +=  0x00d1;
		break;
	    case 0x5e: // capital 'U' with diaeresis
		retval +=  0x00dc;
		break;
	    case 0x5f: // section sign
		retval +=  0x00a7;
		break;
	    case 0x60: // inverted question mark
		retval +=  0x00bf;
		break;
	    case 0x7b: // small 'a' with diaeresis
		retval +=  0x00e4;
		break;
	    case 0x7c: // small 'o' with diaeresis
		retval +=  0x00f6;
		break;
	    case 0x7d: // small 'n' with tilde
		retval +=  0x00f1;
		break;
	    case 0x7e: // small 'u' with diaeresis
		retval +=  0x00fc;
		break;
	    case 0x7f: // small 'a' with grave
		retval +=  0x00e0;
		break;
	}
    }
    return retval;
}



/** 
 * convert_to_gsm
 * Diese Methode konvertiert Unicode in den GSM Zeichensatz. 
 * Parameter: QString input
 * Rueckgabewert: QString GSMString
 */
QString BlueConvert::convert_to_gsm (const QString& input)  
{
  QString retval;

  for (uint i = 0; i < input.length(); i++) {
  	const ushort& a = input[i].unicode();
    switch (a) {
    default:
      if (a < 0x0080 && (
				/*
				 * see Unicode table for descriptions
				 */
				a == 0x000A ||
				a == 0x000D ||
				(a >= 0x0020 && a <= 0x0023) ||
				(a >= 0x0025 && a <= 0x003f) ||
				(a >= 0x0041 && a <= 0x005a) ||
				(a >= 0x0061 && a <= 0x007a))) {
	retval += a;
      }
      break;
    case 0x0040: // commercial at
      /*
       * we map this to 0x80 instead of 0x00 to avoid problem with C's char*
       * just make sure you only use the last 7 bits of these values,
       * you must do that anyway
       */
      retval +=  0x80;
      break;
    case 0x00a3: // pound sign
      retval +=  0x01;
      break;
    case 0x0024: // dollar sign
      retval +=  0x02;
      break;
    case 0x00a5: // yen sign
      retval +=  0x03;
      break;
    case 0x00e8: // small 'e' with grave
      retval +=  0x04;
      break;
    case 0x00e9: // small 'e' with acute
      retval +=  0x05;
      break;
    case 0x00f9: // small 'u' with grave
      retval +=  0x06;
      break;
    case 0x00ec: // small 'i' with grave
      retval +=  0x07;
      break;
    case 0x00f2: // small 'o' with grave
      retval +=  0x08;
      break;
    case 0x00c7: // capital 'C' with cedilla
    case 0x00e7: // small 'c' with cedilla
      retval +=  0x09;
      break;
    case 0x00d8: // capital 'O' with stroke
      retval +=  0x0b;
      break;
    case 0x00f8: // small 'o' with stroke
      retval +=  0x0c;
      break;
    case 0x00c5: // capital 'A' with ring above
      retval +=  0x0e;
      break;
    case 0x00e5: // small 'a' with ring above
      retval +=  0x0f;
      break;
    case 0x0394: // capital delta
      retval +=  0x10;
      break;
    case 0x005f: // underscore / low line
      retval +=  0x11;
      break;
    case 0x03a6: // capital phi
      retval +=  0x12;
      break;
    case 0x0393: // capital gamma
      retval +=  0x13;
      break;
    case 0x039b: // capital lambda
      retval +=  0x14;
      break;
    case 0x03a9: // capital omega
      retval +=  0x15;
      break;
    case 0x03a0: // capital pi
      retval +=  0x16;
      break;
    case 0x03a8: // capital psi
      retval +=  0x17;
      break;
    case 0x03a3: // capital sigma
      retval +=  0x18;
      break;
    case 0x0398: // capital theta
      retval +=  0x19;
      break;
    case 0x039e: // capital xi
      retval +=  0x1a;
      break;
    case 0x000c: // form feed
      retval +=  0x1b;
      retval +=  0x0a;
      break;
    case 0x005e: // circumflex accent
      retval +=  0x1b;
      retval +=  0x14;
      break;
    case 0x007b: // left curly bracket
      retval +=  0x1b;
      retval +=  0x28;
      break;
    case 0x007d: // right curly bracket
      retval +=  0x1b;
      retval +=  0x29;
      break;
    case 0x005c: // reverse solidus (back slash)
      retval +=  0x1b;
      retval +=  0x2f;
      break;
    case 0x005b: // left square bracket
      retval +=  0x1b;
      retval +=  0x3c;
      break;
    case 0x007e: // tilde
      retval +=  0x1b;
      retval +=  0x3d;
      break;
    case 0x005d: // right square bracket
      retval +=  0x1b;
      retval +=  0x3e;
      break;
    case 0x007c: // vertical line
      retval +=  0x1b;
      retval +=  0x40;
      break;
    case 0x20ac: // Euro sign
      retval +=  0x1b;
      retval +=  0x65;
      break;
    case 0x00c6: // capital 'AE'
      retval +=  0x1c;
      break;
    case 0x00e6: // small 'ae'
      retval +=  0x1d;
      break;
    case 0x00df: // small sharp s (german)
      retval +=  0x1e;
      break;
    case 0x00c9: // capital 'E' with acute
      retval +=  0x1f;
      break;
    case 0x00a4: // currency sign
    case 0x00b0: // degree sign (not offical, preference of author)
      retval +=  0x24;
      break;
    case 0x00a1: // inverted exclamation mark
      retval +=  0x40;
      break;
    case 0x0391: // capital alpha
      retval +=  0x41;
      break;
    case 0x0392: // capital beta
      retval +=  0x42;
      break;
    case 0x0395: // capital epsilon
      retval +=  0x45;
      break;
    case 0x0397: // capital eta
      retval +=  0x48;
      break;
    case 0x0399: // capital iota
      retval +=  0x49;
      break;
    case 0x039a: // capital kappa
      retval +=  0x4b;
      break;
    case 0x039c: // capital mu
      retval +=  0x4d;
      break;
    case 0x039d: // capital nu
      retval +=  0x4e;
      break;
    case 0x039f: // capital omicron
      retval +=  0x4f;
      break;
    case 0x03a1: // capital rho
      retval +=  0x50;
      break;
    case 0x03a4: // capital tau
      retval +=  0x54;
      break;
    case 0x03a5: // capital upsilon
      retval +=  0x55;
      break;
    case 0x03a7: // capital chi
      retval +=  0x58;
      break;
    case 0x0396: // capital zeta
      retval +=  0x5a;
      break;
    case 0x00c4: // capital 'A' with diaeresis
      retval +=  0x5b;
      break;
    case 0x00d6: // capital 'O' with diaeresis
      retval +=  0x5c;
      break;
    case 0x00d1: // capital 'N' with tilde
      retval +=  0x5d;
      break;
    case 0x00dc: // capital 'U' with diaeresis
      retval +=  0x5e;
      break;
    case 0x00a7: // section sign
      retval +=  0x5f;
      break;
    case 0x00bf: // inverted question mark
      retval +=  0x60;
      break;
    case 0x00e4: // small 'a' with diaeresis
      retval +=  0x7b;
      break;
    case 0x00f6: // small 'o' with diaeresis
      retval +=  0x7c;
      break;
    case 0x00f1: // small 'n' with tilde
      retval +=  0x7d;
      break;
    case 0x00fc: // small 'u' with diaeresis
      retval +=  0x7e;
      break;
    case 0x00e0: // small 'a' with grave
      retval +=  0x7f;
      break;
    }
  }
  return retval;
}



/** 
 * decodeMessage
 * Diese Methode konvertiert den ERICSSON Zeichensatz (SMS) in Unicode.
 * Parameter: QString input
 * Rueckgabewert: QString decodedMessage
 */
QString BlueConvert::decodeMessage(QString encodedMessage)
{
  QString decodedMessage="";
  unsigned char data[281];
  unsigned char dec[161];
  unsigned int i;
  int c=0;
  int offset;
  int first;
  int end;
  
  for (i=0;i<encodedMessage.length();i+=2)
  {
    data[c] = (encodedMessage.mid(i, 2)).toInt(0,16);
    c++;
  }

  offset = -1;
  c = 0;

  for(i=0;i<m_messageLength;i++)
  {
    if (i%8)
    {
      first = data[i-1-offset];
      first >>= 8 - (i % 8);
      end = data[i - offset];
      end <<= (i % 8);
      *(dec + c) = (first | end) & 127;      
    }
    else
    {
      ++offset;
      *(dec + c) = data[i - offset] & 127;
    }

    if (*(dec + c) == 0)
	{
      *(dec + c) = 128;     
	}
    c++;
  }
  
  dec[c] = '\0';

  //Ok, 7bit message decoded, now convert to human readable signs

  for(i=0;i<strlen((char*)dec);i++)
  {
    switch(dec[i])
    {

      case 1:
        decodedMessage.append("");
        break;

      case 2:
        decodedMessage.append("$");
        break;

      case 3:
        decodedMessage.append("");
        break;

      case 4:
        decodedMessage.append("");
        break;

      case 5:
        decodedMessage.append("");
        break;

      case 6:
        decodedMessage.append("");
        break;

      case 7:
        decodedMessage.append("");
        break;

      case 8:
        decodedMessage.append("");
        break;

      case 9:
        decodedMessage.append("");
        break;

      case 10:
      	//decodedMessage.append("\n"); //linefeed
        break;

      case 11:
        decodedMessage.append("");
        break;

      case 12:
        decodedMessage.append("");
        break;

      case 13:
        //decodedMessage.append("\r"); //carriage return
        break;

      case 14:
        decodedMessage.append("");
        break;

      case 15:
        decodedMessage.append("");
        break;

      case 16:
        //captial DELTA
        break;

      case 17:
        decodedMessage.append("_");
        break;

      case 18:
        //captial PHI
        break;

      case 19:
        //captial GAMMA
        break;

      case 20:
        //captial LAMBDA
        break;

      case 21:
        //captial OMEGA
        break;

      case 22:
        //captial PI
        break;

      case 23:
        //captial PSI
        break;

      case 24:
        //captial SIGMA
        break;

      case 25:
        //captial THETA
        break;

      case 26:
        //captial XI
        break;

      case 27:  //escape to extension table
        i++;
        switch (dec[i])
        {
          case 10:
            //form feed
            break;

          case 20:
            decodedMessage.append("^");
            break;

          case 40:
            decodedMessage.append("{");
            break;

          case 41:
            decodedMessage.append("}");
            break;

          case 47:
            decodedMessage.append("\\");
            break;

          case 60:
            decodedMessage.append("[");
            break;

          case 61:
            decodedMessage.append("~");
            break;

          case 62:
            decodedMessage.append("]");
            break;

          case 64:
            decodedMessage.append("|");
            break;

          case 101:
            //EURO SIGN decodedMessage.append("");
            break;
        }        
        break;

      case 28:
        decodedMessage.append("");
        break;

      case 29:
        decodedMessage.append("");
        break;

      case 30:
        decodedMessage.append("");
        break;

      case 31:
        decodedMessage.append("");
        break;

      case 32:
        decodedMessage.append(" ");
        break;

      case 33:
        decodedMessage.append("!");
        break;

      case 34:
        decodedMessage.append("\"");
        break;

      case 35:
        decodedMessage.append("#");
        break;

      case 36:
        //currency sign decodedMessage.append("");
        break;

      case 37:
        decodedMessage.append("%");
        break;

      case 38:
        decodedMessage.append("&");
        break;

      case 39:
        decodedMessage.append("'");
        break;

      case 40:
        decodedMessage.append("(");
        break;

      case 41:
        decodedMessage.append(")");
        break;

      case 42:
        decodedMessage.append("*");
        break;

      case 43:
        decodedMessage.append("+");
        break;

      case 44:
        decodedMessage.append(",");
        break;

      case 45:
        decodedMessage.append("-");
        break;

      case 46:
        decodedMessage.append(".");
        break;

      case 47:
        decodedMessage.append("/");
        break;

      case 48:
        decodedMessage.append("0");
        break;

      case 49:
        decodedMessage.append("1");
        break;

      case 50:
        decodedMessage.append("2");
        break;

      case 51:
        decodedMessage.append("3");
        break;

      case 52:
        decodedMessage.append("4");
        break;

      case 53:
        decodedMessage.append("5");
        break;

      case 54:
        decodedMessage.append("6");
        break;

      case 55:
        decodedMessage.append("7");
        break;

      case 56:
        decodedMessage.append("8");
        break;

      case 57:
        decodedMessage.append("9");
        break;

      case 58:
        decodedMessage.append(":");
        break;

      case 59:
        decodedMessage.append(";");
        break;

      case 60:
        decodedMessage.append("<");
        break;

      case 61:
        decodedMessage.append("=");
        break;

      case 62:
        decodedMessage.append(">");
        break;

      case 63:
        decodedMessage.append("?");
        break;

      case 64:
        decodedMessage.append("");
        break;

      case 65:
        decodedMessage.append("A");
        break;

      case 66:
        decodedMessage.append("B");
        break;

      case 67:
        decodedMessage.append("C");
        break;

      case 68:
        decodedMessage.append("D");
        break;

      case 69:
        decodedMessage.append("E");
        break;

      case 70:
        decodedMessage.append("F");
        break;                                

      case 71:
        decodedMessage.append("G");
        break;

      case 72:
        decodedMessage.append("H");
        break;

      case 73:
        decodedMessage.append("I");
        break;

      case 74:
        decodedMessage.append("J");
        break;

      case 75:
        decodedMessage.append("K");
        break;

      case 76:
        decodedMessage.append("L");
        break;

      case 77:
        decodedMessage.append("M");
        break;

      case 78:
        decodedMessage.append("N");
        break;

      case 79:
        decodedMessage.append("O");
        break;

      case 80:
        decodedMessage.append("P");
        break;

      case 81:
        decodedMessage.append("Q");
        break;

      case 82:
        decodedMessage.append("R");
        break;

      case 83:
        decodedMessage.append("S");
        break;

      case 84:
        decodedMessage.append("T");
        break;

      case 85:
        decodedMessage.append("U");
        break;

      case 86:
        decodedMessage.append("V");
        break;

      case 87:
        decodedMessage.append("W");
        break;

      case 88:
        decodedMessage.append("X");
        break;

      case 89:
        decodedMessage.append("Y");
        break;

      case 90:
        decodedMessage.append("Z");
        break;

      case 91:
        decodedMessage.append("");
        break;

      case 92:
        decodedMessage.append("");
        break;

      case 93:
        decodedMessage.append("");
        break;

      case 94:
        decodedMessage.append("");
        break;

      case 95:
        decodedMessage.append("");
        break;

      case 96:
        decodedMessage.append("");
        break;

      case 97:
        decodedMessage.append("a");
        break;

      case 98:
        decodedMessage.append("b");
        break;

      case 99:
        decodedMessage.append("c");
        break;

      case 100:
        decodedMessage.append("d");
        break;

      case 101:
        decodedMessage.append("e");
        break;

      case 102:
        decodedMessage.append("f");
        break;

      case 103:
        decodedMessage.append("g");
        break;

      case 104:
        decodedMessage.append("h");
        break;

      case 105:
        decodedMessage.append("i");
        break;

      case 106:
        decodedMessage.append("j");
        break;

      case 107:
        decodedMessage.append("k");
        break;

      case 108:
        decodedMessage.append("l");
        break;

      case 109:
        decodedMessage.append("m");
        break;

      case 110:
        decodedMessage.append("n");
        break;

      case 111:
        decodedMessage.append("o");
        break;

      case 112:
        decodedMessage.append("p");
        break;

      case 113:
        decodedMessage.append("q");
        break;

      case 114:
        decodedMessage.append("r");
        break;

      case 115:
        decodedMessage.append("s");
        break;

      case 116:
        decodedMessage.append("t");
        break;

      case 117:
        decodedMessage.append("u");
        break;

      case 118:
        decodedMessage.append("v");
        break;

      case 119:
        decodedMessage.append("w");
        break;

      case 120:
        decodedMessage.append("x");
        break;

      case 121:
        decodedMessage.append("y");
        break;

      case 122:
        decodedMessage.append("z");
        break;

      case 123:
        decodedMessage.append("");
        break;

      case 124:
        decodedMessage.append("");
        break;

      case 125:
        decodedMessage.append("");
        break;

      case 126:
        decodedMessage.append("");
        break;

      case 127:
        decodedMessage.append("");
        break;

      case 128:
        decodedMessage.append("@");
        break;

      default:
        break;
    } 
  }  
  return decodedMessage;
}



/** 
 * encodeMessage
 * Diese Methode konvertiert Unicode in den ERICSSON Zeichensatz (SMS).
 * Parameter: QString input, int Length
 * Rueckgabewert: QString encodedMessage
 */
QString BlueConvert::encodeMessage(QString decodedMessage, int longMessage)
{
  QString encodedMessage ="";
  QString num;
  unsigned char enc[281];
  int first, end;
  unsigned int i;
  
  m_messageLength = decodedMessage.length();
  
  for (i=0;i<decodedMessage.length();i++)
  {
    switch(decodedMessage.at(i).latin1())
    {

      case '':
        enc[i] = 0x01;
        break;

      case '$':
        enc[i] = 0x02;
        break;

      case '':
        enc[i] = 3;
        break;

      case '':
        enc[i] = 4;
        break;

      case '':
        enc[i] = 5;
        break;

      case '':
        enc[i] = 6;
        break;

      case '':
        enc[i] = 7;
        break;

      case '':
        enc[i] = 8;
        break;

      case '':
        enc[i] = 9;
        break;

      case '\n':
        enc[i] = 10;
        break;

      case '':
        enc[i] = 11;
        break;

      case '':
        enc[i] = 12;
        break;

      case '\r':
        enc[i] = 13;
        break;

      case '':
        enc[i] = 14;
        break;

      case '':
        enc[i] = 15;
        break;

      /*case capital DELTA:
        enc[i] = 16;
        break;                    */

      case '_':
        enc[i] = 17;
        break;

        
  /*  GREEK ALPHABET
      case
        enc[i] = 18;
        break;

      case '':
        enc[i] = 19;
        break;

      case '':
        enc[i] = 21;
        break;

      case '':
        enc[i] = 22;
        break;

      case '':
        enc[i] = 23;
        break;

      case '':
        enc[i] = 24;
        break;

      case '':
        enc[i] = 25;
        break;

      case '':
        enc[i] = 26;
        break;
  */
      case '^':
        enc[i] = 27;
        i++;
        enc[i] = 20;
        break;

      case '{':
        enc[i] = 27;
        i++;
        enc[i] = 40;
        break;

      case '}':
        enc[i] = 27;
        i++;
        enc[i] = 41;
        break;

      case '\\':
        enc[i] = 27;
        i++;
        enc[i] = 47;
        break;

      case '[':
        enc[i] = 27;
        i++;
        enc[i] = 60;
        break;

      case '~':
        enc[i] = 27;
        i++;
        enc[i] = 61;
        break;

      case ']':
        enc[i] = 27;
        i++;
        enc[i] = 62;
        break;

      case '|':
        enc[i] = 27;
        i++;
        enc[i] = 64;
        break;

/*EURO SIGN      case '':
        enc[i] = 27;
        i++;
        enc[i] = 40;
        break;   */

      case '':
        enc[i] = 28;
        break;

      case '':
        enc[i] = 29;
        break;

      case '':
        enc[i] = 30;
        break;

      case '':
        enc[i] = 31;
        break;

      case ' ':
        enc[i] = 32;
        break;

      case '!':
        enc[i] = 33;
        break;

      case '"':
        enc[i] = 34;
        break;

      case '#':
        enc[i] = 35;
        break;

      //CURRENCY SIGN


      case '%':
        enc[i] = 37;
        break;

      case '&':
        enc[i] = 38;
        break;

      case '\'':
        enc[i] = 39;
        break;

      case '(':
        enc[i] = 40;
        break;

      case ')':
        enc[i] = 41;
        break;

      case '*':
        enc[i] = 42;
        break;

      case '+':
        enc[i] = 43;
        break;

      case ',':
        enc[i] = 44;
        break;

      case '-':
        enc[i] = 45;
        break;

      case '.':
        enc[i] = 46;
        break;

      case '/':
        enc[i] = 47;
        break;

      case '0':
        enc[i] = 48;
        break;

      case '1':
        enc[i] = 49;
        break;

      case '2':
        enc[i] = 50;
        break;

      case '3':
        enc[i] = 51;
        break;

      case '4':
        enc[i] = 52;
        break;

      case '5':
        enc[i] = 53;
        break;

      case '6':
        enc[i] = 54;
        break;

      case '7':
        enc[i] = 55;
        break;

      case '8':
        enc[i] = 56;
        break;

      case '9':
        enc[i] = 57;
        break;

      case ':':
        enc[i] = 58;
        break;

      case ';':
        enc[i] = 59;
        break;

      case '<':
        enc[i] = 60;
        break;

      case '=':
        enc[i] = 61;
        break;

      case '>':
        enc[i] = 62;
        break;

      case '?':
        enc[i] = 63;
        break;

      case '':
        enc[i] = 64;
        break;

      case 'A':
        enc[i] = 65;
        break;

      case 'B':
        enc[i] = 66;
        break;

      case 'C':
        enc[i] = 67;
        break;

      case 'D':
        enc[i] = 68;
        break;

      case 'E':
        enc[i] = 69;
        break;

      case 'F':
        enc[i] = 70;
        break;

      case 'G':
        enc[i] = 71;
        break;

      case 'H':
        enc[i] = 72;
        break;

      case 'I':
        enc[i] = 73;
        break;

      case 'J':
        enc[i] = 74;
        break;

      case 'K':
        enc[i] = 75;
        break;

      case 'L':
        enc[i] = 76;
        break;

      case 'M':
        enc[i] = 77;
        break;

      case 'N':
        enc[i] = 78;
        break;

      case 'O':
        enc[i] = 79;
        break;

      case 'P':
        enc[i] = 80;
        break;

      case 'Q':
        enc[i] = 81;
        break;

      case 'R':
        enc[i] = 82;
        break;

      case 'S':
        enc[i] = 83;
        break;

      case 'T':
        enc[i] = 84;
        break;

      case 'U':
        enc[i] = 85;
        break;

      case 'V':
        enc[i] = 86;
        break;

      case 'W':
        enc[i] = 87;
        break;

      case 'X':
        enc[i] = 88;
        break;

      case 'Y':
        enc[i] = 89;
        break;

      case 'Z':
        enc[i] = 90;
        break;

      case '':
        enc[i] = 91;
        break;

      case '':
        enc[i] = 92;
        break;

      case '':
        enc[i] = 93;
        break;

      case '':
        enc[i] = 94;
        break;

      case '':
        enc[i] = 95;
        break;

      case '':
        enc[i] = 96;
        break;

      case 'a':
        enc[i] = 97;
        break;

      case 'b':
        enc[i] = 98;
        break;

      case 'c':
        enc[i] = 99;
        break;

      case 'd':
        enc[i] = 100;
        break;

      case 'e':
        enc[i] = 101;
        break;

      case 'f':
        enc[i] = 102;
        break;

      case 'g':
        enc[i] = 103;
        break;

      case 'h':
        enc[i] = 104;
        break;

      case 'i':
        enc[i] = 105;
        break;

      case 'j':
        enc[i] = 106;
        break;

      case 'k':
        enc[i] = 107;
        break;

      case 'l':
        enc[i] = 108;
        break;

      case 'm':
        enc[i] = 109;
        break;

      case 'n':
        enc[i] = 110;
        break;

      case 'o':
        enc[i] = 111;
        break;

      case 'p':
        enc[i] = 112;
        break;

      case 'q':
        enc[i] = 113;
        break;

      case 'r':
        enc[i] = 114;
        break;

      case 's':
        enc[i] = 115;
        break;

      case 't':
        enc[i] = 116;
        break;

      case 'u':
        enc[i] = 117;
        break;

      case 'v':
        enc[i] = 118;
        break;

      case 'w':
        enc[i] = 119;
        break;

      case 'x':
        enc[i] = 120;
        break;

      case 'y':
        enc[i] = 121;
        break;

      case 'z':
        enc[i] = 122;
        break;

      case '':
        enc[i] = 123;
        break;

      case '':
        enc[i] = 124;
        break;

      case '':
        enc[i] = 125;
        break;

      case '':
        enc[i] = 126;
        break;

      case '':
        enc[i] = 127;
        break;      
      
	  case '@':
        enc[i] = 128;
        break;      
        
      default:
        break;
    }
  }

  enc[i] = '\0';
  
  //Now, we have to encode the message to its 7bit representation
  for (i=0; i<strlen((char*)enc); i++)
  {
	if (i==0 && longMessage) // special treatnmen - would "normally" be skipped since %8==7
  	{
   		first=(enc[0]&127)<<1;
   		num.setNum(first, 16);
	    if (first < 16)
  	    encodedMessage.append("0");
    	encodedMessage.append(num);     
  	}
  	else
  	{
	    end = enc[i] & 127;
  		end >>= ((i-longMessage) % 8);
    	first = enc[i+1] & 127;
	    first <<= 7 - ((i-longMessage) % 8);
  	  	first &= 255;
    	num.setNum((first|end), 16);
	    if ((first|end) < 16)
  	    encodedMessage.append("0");
    	encodedMessage.append(num);
	    if (((i-longMessage) % 8) == 6)
  	    i++;
  	}
  } 
  return encodedMessage;
}



