
#include "SerialPort.h"
#include <string>
#include <sstream>
#include <crtdbg.h> // C-Runtime Debug

/////////////////////////////////////////////////

SerialPort::SerialPort()
           :m_handle(INVALID_HANDLE_VALUE),
            m_portNumber(0),
            m_thread(0)
{
}

/////////////////////////////////////////////////

SerialPort::~SerialPort()
{
    close();
}

/////////////////////////////////////////////////

unsigned char SerialPort::getPortNumber() const
{
    return m_portNumber;
}

/////////////////////////////////////////////////

void SerialPort::open(unsigned char port)
{
    unsigned long i = port;
    std::stringstream stream;
    stream << "COM" << i;
    std::string portBez = stream.str();
    
    m_handle = CreateFile(portBez.c_str(),
                          GENERIC_READ | GENERIC_WRITE,
                          0,
                          0,
                          OPEN_EXISTING,
                          FILE_ATTRIBUTE_NORMAL | FILE_FLAG_OVERLAPPED,
                          0);
    
    if(INVALID_HANDLE_VALUE == m_handle)
    {
        throw std::string("Konnte Port nicht ffnen");
    }
    
    COMMTIMEOUTS ct = { 0 };
    BOOL ok = SetCommTimeouts(m_handle, &ct);
    if(FALSE == ok)
    {
        throw std::string("Konnte Timeouts nicht setzen");
    }
    
    DCB dcb = { 0 };
    BOOL dcbOk = GetCommState(m_handle, &dcb);
    if(FALSE == dcbOk)
    {
        throw std::string("Konnte State nicht erfragen");
    }
    
    dcb.BaudRate = CBR_9600;
    dcb.ByteSize = 8;
    dcb.Parity = NOPARITY;
    dcb.StopBits = ONESTOPBIT;
    
    dcbOk = SetCommState(m_handle, &dcb);
    if(FALSE == dcbOk)
    {
        throw std::string("Konnte State nicht setzen");
    }
    
    m_portNumber = port;
}

/////////////////////////////////////////////////

void SerialPort::close()
{
    if(INVALID_HANDLE_VALUE != m_handle)
    {
        CloseHandle(m_handle);
        m_handle = INVALID_HANDLE_VALUE;
    }
}

/////////////////////////////////////////////////

unsigned char SerialPort::readByte()
{
    if(INVALID_HANDLE_VALUE == m_handle)
    {
        throw std::string("readByte : Port nicht offen");
    }
    
    OVERLAPPED ol = { 0 };
    ol.hEvent = CreateEvent(0, FALSE, FALSE, 0);
    
    unsigned char data = 0;
    DWORD bytesRead = 0;
    BOOL result = FALSE;

    result = ReadFile(m_handle,
                      &data,
                      1,
                      &bytesRead,
                      &ol);

    if(FALSE == result)
    {
        DWORD error = GetLastError();
        if(ERROR_IO_PENDING == error)
        {
            WaitForSingleObject(ol.hEvent, INFINITE);
            CloseHandle(ol.hEvent);
        }
        else
        {
            std::stringstream stream;
            stream << "ReadFile misslungen mit code : ";
            stream << error;
            throw stream.str();
        }
    }
    
    return data;
}

/////////////////////////////////////////////////

void SerialPort::writeByte(unsigned char data)
{
    if(INVALID_HANDLE_VALUE == m_handle)
    {
        throw std::string("writeByte : Port nicht offen");
    }
    
    OVERLAPPED ol = { 0 };
    ol.hEvent = CreateEvent(0, FALSE, FALSE, 0);
    
    DWORD bytesWritten = 0;
    BOOL result = FALSE;
    result = WriteFile(m_handle,
        &data,
        1,
        &bytesWritten,
        &ol);
    if(FALSE == result)
    {
        DWORD error = GetLastError();
        if(ERROR_IO_PENDING == error)
        {
            WaitForSingleObject(ol.hEvent, INFINITE);
            CloseHandle(ol.hEvent);
        }
        else
        {
            std::stringstream stream;
            stream << "WriteFile misslungen mit code : ";
            stream << error;
            throw stream.str();
        }
    }
}

/////////////////////////////////////////////////

SerialPort& SerialPort::getSerialPort(unsigned char portNumber)
{
    static SerialPorts theSerialPorts;
    
    SerialPorts::iterator it = theSerialPorts.find(portNumber);
    if(it == theSerialPorts.end())
    {
        theSerialPorts[portNumber] = new SerialPort;
        try
        {
            theSerialPorts[portNumber]->open(portNumber);
        }
        catch(const std::string& error)
        {
            theSerialPorts.erase(portNumber);
            throw error;
        }
    }
    
    return *theSerialPorts[portNumber];
}

/////////////////////////////////////////////////

void SerialPort::addObserver(SerialPortObserver* observer)
{
    CSLock lock(m_cs);
    
    m_observers.insert(observer);
    
    if(0 == m_thread)
    {
        DWORD threadId = 0;
        m_thread = CreateThread(0,
            0,
            readThread,
            this,
            0,
            &threadId);
    }
}

/////////////////////////////////////////////////

void SerialPort::removeObserver(SerialPortObserver* observer)
{
    CSLock lock(m_cs);
    m_observers.erase(observer);
}

/////////////////////////////////////////////////

void SerialPort::readAndDispatch()
{
    while(1)
    {
        unsigned char data = readByte();

        CSLock lock(m_cs);
        
        if(m_observers.empty())
        {
            break;
        }

        Observers::iterator it = m_observers.begin();
        Observers::iterator end = m_observers.end();
        for( ; it != end; ++it)
        {
            SerialPortObserver* observer = (*it);
            observer->newSerialData(*this, data);
        }
    }
}

/////////////////////////////////////////////////

DWORD WINAPI SerialPort::readThread(LPVOID param)
{
    SerialPort* thePort = static_cast<SerialPort*>(param);
    
    _ASSERT(thePort != 0);
    
    thePort->readAndDispatch();
    
    return 0;
}

/////////////////////////////////////////////////
